/*
 * baslist.c -- list a BASIC program from a binary format (.bba) file.
 *
 * Author: George Phillips <phillips@cs.ubc.ca>
 */

#include <stdio.h>

char* token[] = {
	"END",
	"FOR", "RESET", "SET", "CLS", "CMD", "RANDOM", "NEXT", "DATA", "INPUT",
	"DIM", "READ", "LET", "GOTO", "RUN", "IF", "RESTORE", "GOSUB", "RETURN",
	"REM", "STOP", "ELSE", "TRON", "TROFF", "DEFSTR", "DEFINT", "DEFSNG",
	"DEFDBL", "LINE", "EDIT", "ERROR", "RESUME", "OUT", "ON", "OPEN",
	"FIELD", "GET", "PUT", "CLOSE", "LOAD", "MERGE", "NAME", "KILL", "LSET",
	"RSET", "SAVE", "SYSTEM", "LPRINT", "DEF", "POKE", "PRINT", "CONT",
	"LIST", "LLIST", "DELETE", "AUTO", "CLEAR", "CLOAD", "CSAVE", "NEW",
	"TAB(", "TO", "FN", "USING", "VARPTR", "USR", "ERL", "ERR", "STRING$",
	"INSTR", "POINT", "TIME$", "MEM", "INKEY$", "THEN", "NOT", "STEP",
	"+", "-", "*", "/", "[", "AND", "OR", ">", "=", "<", "SGN", "INT",
	"ABS", "FRE", "INP", "POS", "SQR", "RND", "LOG", "EXP", "COS", "SIN",
	"TAN", "ATN", "PEEK", "CVI", "CVS", "CVD", "EOF", "LOC", "LOF", "MKI$",
	"MKS$", "MKD$", "CINT", "CSNG", "CDBL", "FIX", "LEN", "STR$", "VAL",
	"ASC", "CHR$", "LEFT$", "RIGHT$", "MID$" };

#define ntok	(sizeof(token) / sizeof(char*))
#define REM		(147)
#define DATA	(136)
#define REMQUOT	(251)
#define ELSE	(149)

main(int argc, char* argv[])
{
	int		i;
	int		ch;
	FILE*	fp;
	int		tstate;

	if (argc != 2) {
		fprintf(stderr, "usage: baslist file.bba\n");
		exit(1);
	}

	if ((fp = fopen(argv[1], "rb")) == NULL) {
		fprintf(stderr, "can't open '%s'\n", argv[1]);
		exit(1);
	}

	if (fgetc(fp) != 255) {
		fprintf(stderr, "missing magic # of 255 -- not a BASIC file.\n");
		exit(1);
	}

	for (;;) {
		if ((ch = fgetc(fp)) == EOF || (i = fgetc(fp)) == EOF) {
			fprintf(stderr, "EOF in next line#\n");
			exit(2);
		}
		if (i == 0 && ch == 0)
			break;

		if ((ch = fgetc(fp)) == EOF || (i = fgetc(fp)) == EOF) {
			fprintf(stderr, "EOF in line#\n");
			exit(2);
		}
		printf("%d ", ch + i * 256);
		tstate = 0;
		while ((ch = fgetc(fp)) != EOF && ch != 0) {
			if (ch == ':' && tstate == 0) {
				tstate = 3;
				continue;
			}
			else if (ch == REM && tstate == 3) {
				tstate = 4;
				continue;
			}
			else if (ch == REMQUOT && tstate == 4) {
				putchar('\'');
				tstate = 2;
				continue;
			}
			else if (ch == ELSE && tstate == 3) {
				printf("ELSE");
				tstate = 0;
				continue;
			}

			if (tstate > 2) {
				putchar(':');
				if (tstate > 3)
					printf("REM");
				tstate = 0;
			}

			switch (tstate) {
			case 0:	/* just cruising over the line */
				if (ch > 127 && ch < 128 + ntok)
					printf("%s", token[ch - 128]);
				else
					putchar(ch);

				if (ch == DATA || ch == REM)
					tstate = 2;
				else if (ch == '"')
					tstate = 1;
				break;
			case 1: /* inside a quoted string */
				if (ch == '\r')
					printf("\\n");
				else if (ch == '\\')
					printf("\\%03o", ch);
				else if (ch >= ' ' && ch < 128)
					putchar(ch);
				else
					printf("\\%03o", ch);
				if (ch == '"')
					tstate = 0;
				break;
			case 2: /* dumping from a REM or DATA statement to line's end */
				putchar(ch);
				break;
			}
		}

		if (tstate > 2) {
			putchar(':');
			if (tstate > 3)
				printf("REM");
			tstate = 0;
		}

		printf("\n");
		if (ch == EOF) {
			fprintf(stderr, "EOF in line\n");
			exit(2);
		}
	}

	fclose(fp);
	exit(0);
}
