////////////////////////////////////////////////////////////////////////////////
// SID6581.cpp -- this file is part of the Emulator Developers Kit
// available at http://ourworld.compuserve.com/homepages/pc64/develop.htm

RegisterPersistentClass(SID6581);


////////////////////////////////////////////////////////////////////////////////
// initialisation

void SID6581::DoInit() {

  // initialize base class
  Chip::DoInit();

  // initialize components
  Reset.Init("Reset", this);
  Reset.SetOnHigh((pfn)OnReset);
  Reset.SetOnLow((pfn)OnReset);
}


////////////////////////////////////////////////////////////////////////////////
// frequency registers

void SID6581::Write1FreqLow(byte bValue) {
  aiFreq[0] = (aiFreq[0] & 0xFF00) | bValue;
}

void SID6581::Write2FreqLow(byte bValue) {
  aiFreq[1] = (aiFreq[1] & 0xFF00) | bValue;
}

void SID6581::Write3FreqLow(byte bValue) {
  aiFreq[2] = (aiFreq[2] & 0xFF00) | bValue;
}

void SID6581::Write1FreqHigh(byte bValue) {
  aiFreq[0] = (aiFreq[0] & 0x00FF) | (bValue << 8);
}

void SID6581::Write2FreqHigh(byte bValue) {
  aiFreq[1] = (aiFreq[1] & 0x00FF) | (bValue << 8);
}

void SID6581::Write3FreqHigh(byte bValue) {
  aiFreq[2] = (aiFreq[2] & 0x00FF) | (bValue << 8);
}


////////////////////////////////////////////////////////////////////////////////
// pulse width registers

void SID6581::Write1PulseWidthLow(byte bValue) {
  aiPulseWidth[0] = (aiPulseWidth[0] & 0x0F00) | bValue;
}

void SID6581::Write2PulseWidthLow(byte bValue) {
  aiPulseWidth[1] = (aiPulseWidth[1] & 0x0F00) | bValue;
}

void SID6581::Write3PulseWidthLow(byte bValue) {
  aiPulseWidth[2] = (aiPulseWidth[2] & 0x0F00) | bValue;
}

void SID6581::Write1PulseWidthHigh(byte bValue) {
  aiPulseWidth[0] = (aiPulseWidth[0] & 0x00FF) | ((bValue << 8) & 0x0F00) ;
}

void SID6581::Write2PulseWidthHigh(byte bValue) {
  aiPulseWidth[1] = (aiPulseWidth[1] & 0x00FF) | ((bValue << 8) & 0x0F00) ;
}

void SID6581::Write3PulseWidthHigh(byte bValue) {
  aiPulseWidth[2] = (aiPulseWidth[2] & 0x00FF) | ((bValue << 8) & 0x0F00);
}


////////////////////////////////////////////////////////////////////////////////
// start values for envelope counters

const int SID6581::gaiStart[16] = {
  9, 32, 63, 95, 149, 220, 267, 313, 392, 977, 1954, 3126, 3906, 11720, 19532, 31251
};


////////////////////////////////////////////////////////////////////////////////
// envelope registers

void SID6581::Write1AttackDecay(byte bValue) {
  aiAttack[0] = gaiStart[bValue >> 4];
  aiDecay[0] = gaiStart[bValue & 15];
}

void SID6581::Write2AttackDecay(byte bValue) {
  aiAttack[1] = gaiStart[bValue >> 4];
  aiDecay[1] = gaiStart[bValue & 15];
}

void SID6581::Write3AttackDecay(byte bValue) {
  aiAttack[2] = gaiStart[bValue >> 4];
  aiDecay[2] = gaiStart[bValue & 15];
}

void SID6581::Write1SustainRelease(byte bValue) {
  aiSustain[0] = (bValue & 0xF0) | (bValue >> 4);
  aiRelease[0] = gaiStart[bValue & 15];
}

void SID6581::Write2SustainRelease(byte bValue) {
  aiSustain[1] = (bValue & 0xF0) | (bValue >> 4);
  aiRelease[1] = gaiStart[bValue & 15];
}

void SID6581::Write3SustainRelease(byte bValue) {
  aiSustain[2] = (bValue & 0xF0) | (bValue >> 4);
  aiRelease[2] = gaiStart[bValue & 15];
}


////////////////////////////////////////////////////////////////////////////////
// control registers

void SID6581::Write1Control(byte bValue) {

  // reset counter
  if ((bValue & fTest) != 0) {
    aiCounter[0] = 0;
    aiNoise[0] = -1;
  }

  // key bit changed?
  if (((bValue ^ abControl[0]) & fKey) != 0) {

    // key bit set
    if ((bValue & fKey) != 0) {
      afEnvIncrement[0] = true;
      aiEnvCounter[0] = aiAttack[0];

    // key bit cleared
    } else {
      afEnvIncrement[0] = false;
    }

  }
  abControl[0] = bValue;
}

void SID6581::Write2Control(byte bValue) {

  // reset counter
  if ((bValue & fTest) != 0) {
    aiCounter[1] = 0;
    aiNoise[1] = -1;
  }

  // key bit changed?
  if (((bValue ^ abControl[1]) & fKey) != 0) {

    // key bit set
    if ((bValue & fKey) != 0) {
      afEnvIncrement[1] = true;
      aiEnvCounter[1] = aiAttack[1];

    // key bit cleared
    } else {
      afEnvIncrement[1] = false;
    }

  }
  abControl[1] = bValue;
}

void SID6581::Write3Control(byte bValue) {

  // reset counter
  if ((bValue & fTest) != 0) {
    aiCounter[2] = 0;
    aiNoise[2] = -1;
  }

  // key bit changed?
  if (((bValue ^ abControl[2]) & fKey) != 0) {

    // key bit set
    if ((bValue & fKey) != 0) {
      afEnvIncrement[2] = true;
      aiEnvCounter[2] = aiAttack[2];

    // key bit cleared
    } else {
      afEnvIncrement[2] = false;
    }

  }
  abControl[2] = bValue;
}


////////////////////////////////////////////////////////////////////////////////
// filter and volume registers

void SID6581::WriteResFilt(byte bValue) {
  bResFilt = bValue;
}

void SID6581::WriteModeVol(byte bValue) {
  bModeVol = bValue;

  // 15 volume, 3 voices, 255 wave, 255 envelope
  iVolume = (bValue & 0x0F) * (0xFFFFFFFF / (15 * 3 * 255 * 255));
}  


////////////////////////////////////////////////////////////////////////////////
// read registers

byte SID6581::Read3Output() {

  // dummy random generator for NoSound
  if (fNoSoundMode) {
    abOutput[2] = (byte)(abOutput[2] * 13 + 1);
  }
  return abOutput[2];
}

byte SID6581::Read3Envelope() {
  return (byte)aiEnvelope[2];
}


////////////////////////////////////////////////////////////////////////////////
// SID main function
// maximal jump over 256 clocks, means use a minimal sampling rate of 4 KHz

#pragma optimize("t", on)
word SID6581::GetNextSample(int iClocksSinceLastSample) {

  // for the three channels...
  for (int i = 0; i < 3; i++) {
    static const int aiOtherChannel[3] = { 2, 0, 1 };


    ////////////////////////////////////////////////////////////
    // frequency counter

    // save old value to detect bit changes
    aiOldCounter[i] = aiCounter[i];

    // don't count as long as the test bit is set
    if ((abControl[i] & fTest) == 0) {

      // 24 bit Bresenham counter: o2 * Freq / 2^24
      aiCounter[i] += iClocksSinceLastSample * aiFreq[i];

      // sync if bit 23 of the other counter has gone high
      // - iClocksSinceLastSample may not be larger than 256 clocks
      // - channel 1 sync is delayed by iClocksSinceLastSample in this implementation
      if ((abControl[i] & fSync) != 0) {
        if ((aiOldCounter[aiOtherChannel[i]] & (1 << 23)) == 0 && (aiCounter[aiOtherChannel[i]] & (1 << 23)) != 0) {
          aiCounter[i] = 0;
        }
      }

    }

    // shift 23 bit noise generator when counter bit 20 has changed
    if (((aiCounter[i] ^ aiOldCounter[i]) & (1 << 20)) != 0) {
      aiNoise[i] = (aiNoise[i] << 1) | (((aiNoise[i] >> 22) ^ (aiNoise[i] >> 17)) & 1);
    }


    ////////////////////////////////////////////////////////////
    // single waveforms

    switch (abControl[i] & 0xF0) {

    case fTriangle:
      abOutput[i] = (byte)(aiCounter[i] >> 15);
      if ((aiCounter[i] & (1 << 23)) != 0) {
        abOutput[i] ^= 0xFF;
      }

      // ring modulation
      if ((abControl[i] & fRing) != 0) {
        if ((aiCounter[aiOtherChannel[i]] & (1 << 23)) == 0) {
          abOutput[i] ^= 0xFF;
        }
      }

      break;

    case fSawtooth:
      abOutput[i] = (byte)(aiCounter[i] >> 16);
      break;

    case fPulse:
      if (((aiCounter[i] >> 12) & 4095) < aiPulseWidth[i]) {
        abOutput[i] = 0;
      } else {
        abOutput[i] = 255;
      }
      break;

    case fNoise:
      abOutput[i] = (byte)((aiNoise[i] >> (22 - 7)) & 0x80 | (aiNoise[i] >> (20 - 6)) & 0x40 | (aiNoise[i] >> (16 - 5)) & 0x20 | (aiNoise[i] >> (13 - 4)) & 0x10 | (aiNoise[i] >> (11 - 3)) & 0x08 | (aiNoise[i] >> (7 - 2)) & 0x04 | (aiNoise[i] >> (4 - 1)) & 0x02 | (aiNoise[i] >> (2 - 0)) & 0x01);
      break;


    ////////////////////////////////////////////////////////////
    // combined waveforms (too complicated, just using tables)

    case fTriangle | fSawtooth:
      {
        static const byte abTriangleAndSawtooth[128] = {
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 28,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 56, 60,  0,
        };
        abOutput[i] = abTriangleAndSawtooth[(aiCounter[i] >> 16) & 127];
      }
      break;

    case fTriangle | fPulse:
      {
        static const byte abTriangleAndPulse[256] = {
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,112,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,176,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,192,  0,
            0,  0,  0,  0,  0,128,224,  0,128,128,240,128,240,248,254,255,
          252,248,192,240,128,128,  0,232,128,128,  0,128,  0,  0,  0,216,
          128,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,184,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,120,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0
        };
        if (((aiCounter[i] >> 12) & 4095) < aiPulseWidth[i]) {
          abOutput[i] = 0;
        } else {
          abOutput[i] = abTriangleAndPulse[(aiCounter[i] >> 16) & 255];
        }
      }
      break;

    case fSawtooth | fPulse:
      {
        static const byte abSawtoothAndPulse[128] = {
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
            0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 64,  0,
            0,  0,  0,  0,  0,  0, 64,  0,  0,  0, 96,  0,112,112,127,  0
        };
        if (((aiCounter[i] >> 12) & 2047) < aiPulseWidth[i]) {
          abOutput[i] = 0;
        } else {
          abOutput[i] = abSawtoothAndPulse[(aiCounter[i] >> 16) & 127];
        }
      }
      break;

    default:
      abOutput[i] = 0;
    }


    ////////////////////////////////////////////////////////////
    // envelope

    // exponential curve for decay and release, needs further research
    static const byte abEnvCurveFactor[256] = {
      0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x10,
      0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x04,0x04,0x04,0x04,
      0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,
      0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
      0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
      0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,0x02,
      0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
      0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
      0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
      0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
      0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
      0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
      0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
      0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
      0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
      0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01
    };

    aiEnvCounter[i] -= iClocksSinceLastSample;

    // attack
    if (afEnvIncrement[i]) {
      while (aiEnvCounter[i] < 0) {
        aiEnvCounter[i] += aiAttack[i];
        aiEnvelope[i]++;
      }
      if (aiEnvelope[i] >= 0xFF) {
        aiEnvelope[i] = 0xFF;
        afEnvIncrement[i] = false;
      }

    // decay
    } else if ((abControl[i] & fKey) != 0) {
      while (aiEnvCounter[i] < 0) {
        aiEnvCounter[i] += aiDecay[i] * abEnvCurveFactor[aiEnvelope[i]];

        // sustain
        if ((aiEnvelope[i] != 0) && (aiEnvelope[i] != aiSustain[i])) {
          aiEnvelope[i]--;
        }
      }

    // release
    } else {
      while (aiEnvCounter[i] < 0) {
        aiEnvCounter[i] += aiRelease[i] * abEnvCurveFactor[aiEnvelope[i]];
        if (aiEnvelope[i] != 0) {
          aiEnvelope[i]--;
        }
      }
    }
  }
        

  ////////////////////////////////////////////////////////////
  // sum up waves

  dword dwSum = abOutput[0] * aiEnvelope[0] + abOutput[1] * aiEnvelope[1];
  if ((bModeVol & fMute3) == 0) {
    dwSum += abOutput[2] * aiEnvelope[2];
  }
  dwSum = ((3 * 255 * 255) - dwSum) * iVolume;

  // scale to 16 bit
  return (word)(dwSum >> 16);
}
