/* TODO:
- implement DMA/RDY delay on writes
- test DMA
- test BRK+IRQ+NMI+DMA
- when is NMI flip flop reset?
- Reset must be 6 cycles without writing to stack
- does Reset change SP?
- add persistence
*/

////////////////////////////////////////////////////////////////////////////////
// CPU65xx.cpp -- this file is part of the Emulator Developers Kit
// available at http://ourworld.compuserve.com/homepages/pc64/develop.htm
//
// ANEb: A = X = (A | $EE) & #byte
// On a real C64, when there is a DMA between fetching the opcode
// and the byte, '(A | $EE)' is replaced by $EE/EF/FE/FF randomly.
// This isn't emulated.
// 
// LASay:
// On a real C64, when there is a DMA between fetching the opcode
// and the operand, the result may be $00 or $20. This isn't
// emulated.
//
// SHAay, SHAiy, SHXay, SHYax and SHSay:
// On a real C64, when there is a DMA between fetching the opcode
// and the operand, the part '& (#hh + 1)' may be omitted. This isn't
// emulated.
// 
// SHAay, SHAiy, SHXay, SHYax and SHSay:
// On a real C64, when the target address needs a fixup, those
// commands are writing to different locations, depending on the
// data written. This isn't emulated.
// 
// All other things should work 100% the same as on my 6510. The
// 6502 in the floppy hasn't been tested yet.

RegisterPersistentClass(CPU65xx);


////////////////////////////////////////////////////////////////////////////////
// build the address tables at run time because of Visual C++ 4.2 limitation

static pfn gpfnReset;
#define giRDY 4
#define giNMI 2
#define giIRQ 1
static pfn gapfnInterrupt[8];
static pfn gapfnCommand[256 * 2];
static byte* gpbIndexToProc;

static struct OnLoad {
  byte* pb;
  pfn GetNextAddress() {
    union u {
      int i;
      pfn pfn;
    } u;
    u.i = *(int*)((pb += 5) - 4) + 4;
    return u.pfn;
  }
  OnLoad() {
    extern void CPU65xx_Asm();
    pb = (byte*)CPU65xx_Asm;
    gpfnReset = GetNextAddress();
    gapfnInterrupt[0] = GetNextAddress();
    gapfnInterrupt[1] = GetNextAddress();
    gapfnInterrupt[2] = gapfnInterrupt[3] = GetNextAddress();
    gapfnInterrupt[4] = GetNextAddress();
    for (int i = 5; i < 8; i++) {
      gapfnInterrupt[i] = gapfnInterrupt[4];
    }
    for (i = 0; i < 256 * 2; i++) {
      gapfnCommand[i] = GetNextAddress();
    }
    gpbIndexToProc = pb;
    #ifdef _DEBUG
      if (((int)gapfnCommand & 7) != 0) {
        trace("gapfnCommand is not aligned to 8 bytes!");
      }
      i = 0;
      while (*(int*)(pb + 1) != 0) {
        int iIndex = *(int*)(*(int*)(pb + 1));
        if ((iIndex & 0x0000FFFF) != i) {
          error("CPU65xx proc index entry %08X is %08X!", i, iIndex);
        }
        i++;
        pb += 5;
      }
    #endif
  }
} OnLoad;


////////////////////////////////////////////////////////////////////////////////
// MSVC4 bug: calling member functions from inline assembler is not possible

static void(CPU::* CPU_SetBusy)();
static void(CPU::* CPU_SetIdle)();


////////////////////////////////////////////////////////////////////////////////
// initialisation

void CPU65xx::DoInit() {

  // initialize the base class
  CPU16x4k::DoInit();

  // initialize function pointers for inline asm
  CPU_SetBusy = SetBusy;
  CPU_SetIdle = SetIdle;

  // set function for OnClock
  SetOnClock(gpfnReset);
  SetBusy();

  // initialize components
  Reset.Init("Reset", this);
  Reset.SetOnHigh((pfn)OnResetHigh);
  Reset.SetOnLow((pfn)OnResetLow);
  IRQ.Init("IRQ", this);
  IRQ.SetOnHigh((pfn)OnIRQHigh);
  IRQ.SetOnLow((pfn)OnIRQLow);
  NMI.Init("NMI", this);
  NMI.SetOnLow((pfn)OnNMILow);
  RDY.Init("RDY", this);
  RDY.SetOnHigh((pfn)OnRDYHigh);
  RDY.SetOnLow((pfn)OnRDYLow);

  // set pointer to Command
  pfnInterrupt = gapfnInterrupt[iInterrupts];
}


///////////////////////////////////////////////////////////////////////////////
// To allow the CPU65xx object to store itself to disk, pointers to code
// must be resolved into a number, called an index. The index is the first
// DWORD after the procedure entry. That's why all jumps to indexed procedures
// must go to Name+4.

// TODO: change to (byte*)pfn[-1] == 0 for delayed DMA on writes

#define proc(Name) Name:
#define index(Number) __asm _emit ((Number)) & 0xFF __asm _emit ((Number) >> 8) & 0xFF __asm _emit ((Number) >> 16) & 0xFF __asm _emit ((Number) >> 24) & 0xFF
#define endp

#define giDelayRDY 0x80000000

inline int GetIndexFromProc(pfn pfnProc) {
  union u {
    pfn pfn;
    int* pi;
  } u;
  u.pfn = pfnProc;
  return u.pi[-1];
}

inline pfn GetProcFromIndex(int iIndex) {
  union u {
    byte* pb;
    pfn pfn;
  } u;
  u.pb = gpbIndexToProc + (iIndex & 0x0000FFFF) * 5 + 1;
  return u.pfn;
}


////////////////////////////////////////////////////////////////////////////////
// adjust PC pointer after memory mapping has changed

global void CPU65xx::AdjustPCPtr() {
  int iPC = pbPCPtr - pbPCBase;
  pbPCBase = apbReadMapping[iPC >> 12];
  pbPCPtr = pbPCBase + iPC;
}


////////////////////////////////////////////////////////////////////////////////
// Reset line

void CPU65xx::OnResetLow() {
  if (IsBusy()) {
    SetIdle();
  }
}

void CPU65xx::OnResetHigh() {
  if (RDY.IsInputHigh()) {
    SetBusy();
  }
  SetOnClock(gpfnReset);
}


////////////////////////////////////////////////////////////////////////////////
// IRQ line

void CPU65xx::OnIRQLow() {
  bIRQ = 0x00;
  if ((bFlags & 0x04) == 0 && (iInterrupts & giIRQ) == 0) {
    iInterrupts |= giIRQ;
    pfnInterrupt = gapfnInterrupt[iInterrupts];
    iPostponeIRQ = pClock->GetClocks() + 1;
  }
}

void CPU65xx::OnIRQHigh() {
  bIRQ = 0x04;
  iInterrupts &= ~giIRQ;
  pfnInterrupt = gapfnInterrupt[iInterrupts];
}


////////////////////////////////////////////////////////////////////////////////
// NMI line

void CPU65xx::OnNMILow() {
  if ((iInterrupts & giNMI) == 0) {
    iInterrupts |= giNMI;
    pfnInterrupt = gapfnInterrupt[iInterrupts];
    iPostponeNMI = pClock->GetClocks() + 1;
  }
}


////////////////////////////////////////////////////////////////////////////////
// RDY line

void CPU65xx::OnRDYLow() {
  assert((iInterrupts & giRDY) == 0);
  assert(pfnInterrupt == gapfnInterrupt[iInterrupts]);
  assert(IsBusy() == Reset.IsInputHigh());
  if (IsBusy()) {
    if ((GetIndexFromProc(GetOnClock()) & giDelayRDY) != 0) {
      iInterrupts |= giRDY;
      if (GetOnClock() == pfnInterrupt) {
        SetOnClock(gapfnInterrupt[iInterrupts]);
      }
      pfnInterrupt = gapfnInterrupt[iInterrupts];
    } else {
      SetIdle();
    }
  }
}

void CPU65xx::OnRDYHigh() {
  assert((iInterrupts & giRDY) == 0);  // ???
  assert(pfnInterrupt == gapfnInterrupt[iInterrupts]);
  if (Reset.IsInputHigh()) {
    if ((iInterrupts & giRDY) != 0) {  // ???
      assert(IsBusy());
      iInterrupts &= ~giRDY;
      if (GetOnClock() == pfnInterrupt) {
        SetOnClock(gapfnInterrupt[iInterrupts]);
      }
      pfnInterrupt = gapfnInterrupt[iInterrupts];
    } else {
      assert(IsIdle());
      SetBusy();
    }
  }
}


////////////////////////////////////////////////////////////////////////////////
// CPU65xx main implementation in inline assembler with Pentium pairing

#pragma optimize("g", off)
__declspec(naked) void CPU65xx_Asm() { __asm {


////////////////////////////////////////////////////////////////////////////////
// This is an address table because Visual C++ 4.2 doesn't know the dd directive
// and in _emit(), no calculations with offsets are allowed. As the addresses
// are misaligned (5 bytes each), they are transferred to gapfnInterrupt and
// gapfnCommand at load time.

  mov EAX,offset Reset1
  mov EAX,offset Command
  mov EAX,offset IRQ1
  mov EAX,offset NMI1
  mov EAX,offset RDY

  mov EAX,offset BRKn2  // 00
  mov EAX,offset BRKn2
  mov EAX,offset RWix2  // 01
  mov EAX,offset ORA16
  mov EAX,offset HLTn2  // 02
  mov EAX,offset HLTn2
  mov EAX,offset Mix2   // 03
  mov EAX,offset ASO16
  mov EAX,offset RWz2   // 04
  mov EAX,offset NOP8
  mov EAX,offset RWz2   // 05
  mov EAX,offset ORA8
  mov EAX,offset Mz2    // 06
  mov EAX,offset ASL8
  mov EAX,offset Mz2    // 07
  mov EAX,offset ASO8
  mov EAX,offset PHx2   // 08
  mov EAX,offset PHPn3
  mov EAX,offset ORAb   // 09
  mov EAX,offset ORAb
  mov EAX,offset ASLn   // 0A
  mov EAX,offset ASLn
  mov EAX,offset ANCb   // 0B
  mov EAX,offset ANCb
  mov EAX,offset RWa2   // 0C
  mov EAX,offset NOP16
  mov EAX,offset RWa2   // 0D
  mov EAX,offset ORA16
  mov EAX,offset Ma2    // 0E
  mov EAX,offset ASL16
  mov EAX,offset Ma2    // 0F
  mov EAX,offset ASO16
  mov EAX,offset BPLr2  // 10
  mov EAX,offset BPLr2
  mov EAX,offset Riy2   // 11
  mov EAX,offset ORA16
  mov EAX,offset HLTn2  // 12
  mov EAX,offset HLTn2
  mov EAX,offset Miy2   // 13
  mov EAX,offset ASO16
  mov EAX,offset RWzx2  // 14
  mov EAX,offset NOP8
  mov EAX,offset RWzx2  // 15
  mov EAX,offset ORA8
  mov EAX,offset Mzx2   // 16
  mov EAX,offset ASL8
  mov EAX,offset Mzx2   // 17
  mov EAX,offset ASO8
  mov EAX,offset CLCn   // 18
  mov EAX,offset CLCn
  mov EAX,offset Ray2   // 19
  mov EAX,offset ORA16
  mov EAX,offset NOPn   // 1A
  mov EAX,offset NOPn
  mov EAX,offset May2   // 1B
  mov EAX,offset ASO16
  mov EAX,offset Rax2   // 1C
  mov EAX,offset NOP16
  mov EAX,offset Rax2   // 1D
  mov EAX,offset ORA16
  mov EAX,offset Max2   // 1E
  mov EAX,offset ASL16
  mov EAX,offset Max2   // 1F
  mov EAX,offset ASO16
  mov EAX,offset JSRw2  // 20
  mov EAX,offset JSRw2
  mov EAX,offset RWix2  // 21
  mov EAX,offset AND16
  mov EAX,offset HLTn2  // 22
  mov EAX,offset HLTn2
  mov EAX,offset Mix2   // 23
  mov EAX,offset RLA16
  mov EAX,offset RWz2   // 24
  mov EAX,offset BIT8
  mov EAX,offset RWz2   // 25
  mov EAX,offset AND8
  mov EAX,offset Mz2    // 26
  mov EAX,offset ROL8
  mov EAX,offset Mz2    // 27
  mov EAX,offset RLA8
  mov EAX,offset PLx2   // 28
  mov EAX,offset PLPn4
  mov EAX,offset ANDb   // 29
  mov EAX,offset ANDb
  mov EAX,offset ROLn   // 2A
  mov EAX,offset ROLn
  mov EAX,offset ANCb   // 2B
  mov EAX,offset ANCb
  mov EAX,offset RWa2   // 2C
  mov EAX,offset BIT16
  mov EAX,offset RWa2   // 2D
  mov EAX,offset AND16
  mov EAX,offset Ma2    // 2E
  mov EAX,offset ROL16
  mov EAX,offset Ma2    // 2F
  mov EAX,offset RLA16
  mov EAX,offset BMIr2  // 30
  mov EAX,offset BMIr2
  mov EAX,offset Riy2   // 31
  mov EAX,offset AND16
  mov EAX,offset HLTn2  // 32
  mov EAX,offset HLTn2
  mov EAX,offset Miy2   // 33
  mov EAX,offset RLA16
  mov EAX,offset RWzx2  // 34
  mov EAX,offset NOP8
  mov EAX,offset RWzx2  // 35
  mov EAX,offset AND8
  mov EAX,offset Mzx2   // 36
  mov EAX,offset ROL8
  mov EAX,offset Mzx2   // 37
  mov EAX,offset RLA8
  mov EAX,offset SECn   // 38
  mov EAX,offset SECn
  mov EAX,offset Ray2   // 39
  mov EAX,offset AND16
  mov EAX,offset NOPn   // 3A
  mov EAX,offset NOPn
  mov EAX,offset May2   // 3B
  mov EAX,offset RLA16
  mov EAX,offset Rax2   // 3C
  mov EAX,offset NOP16
  mov EAX,offset Rax2   // 3D
  mov EAX,offset AND16
  mov EAX,offset Max2   // 3E
  mov EAX,offset ROL16
  mov EAX,offset Max2   // 3F
  mov EAX,offset RLA16
  mov EAX,offset RTIn2  // 40
  mov EAX,offset RTIn2
  mov EAX,offset RWix2  // 41
  mov EAX,offset EOR16
  mov EAX,offset HLTn2  // 42
  mov EAX,offset HLTn2
  mov EAX,offset Mix2   // 43
  mov EAX,offset LSE16
  mov EAX,offset RWz2   // 44
  mov EAX,offset NOP8
  mov EAX,offset RWz2   // 45
  mov EAX,offset EOR8
  mov EAX,offset Mz2    // 46
  mov EAX,offset LSR8
  mov EAX,offset Mz2    // 47
  mov EAX,offset LSE8
  mov EAX,offset PHx2   // 48
  mov EAX,offset PHAn3
  mov EAX,offset EORb   // 49
  mov EAX,offset EORb
  mov EAX,offset LSRn   // 4A
  mov EAX,offset LSRn
  mov EAX,offset ALRb   // 4B
  mov EAX,offset ALRb
  mov EAX,offset JMPw2  // 4C
  mov EAX,offset JMPw2
  mov EAX,offset RWa2   // 4D
  mov EAX,offset EOR16
  mov EAX,offset Ma2    // 4E
  mov EAX,offset LSR16
  mov EAX,offset Ma2    // 4F
  mov EAX,offset LSE16
  mov EAX,offset BVCr2  // 50
  mov EAX,offset BVCr2
  mov EAX,offset Riy2   // 51
  mov EAX,offset EOR16
  mov EAX,offset HLTn2  // 52
  mov EAX,offset HLTn2
  mov EAX,offset Miy2   // 53
  mov EAX,offset LSE16
  mov EAX,offset RWzx2  // 54
  mov EAX,offset NOP8
  mov EAX,offset RWzx2  // 55
  mov EAX,offset EOR8
  mov EAX,offset Mzx2   // 56
  mov EAX,offset LSR8
  mov EAX,offset Mzx2   // 57
  mov EAX,offset LSE8
  mov EAX,offset CLIn   // 58
  mov EAX,offset CLIn
  mov EAX,offset Ray2   // 59
  mov EAX,offset EOR16
  mov EAX,offset NOPn   // 5A
  mov EAX,offset NOPn
  mov EAX,offset May2   // 5B
  mov EAX,offset LSE16
  mov EAX,offset Rax2   // 5C
  mov EAX,offset NOP16
  mov EAX,offset Rax2   // 5D
  mov EAX,offset EOR16
  mov EAX,offset Max2   // 5E
  mov EAX,offset LSR16
  mov EAX,offset Max2   // 5F
  mov EAX,offset LSE16
  mov EAX,offset RTSn2  // 60
  mov EAX,offset RTSn2
  mov EAX,offset RWix2  // 61
  mov EAX,offset ADC16
  mov EAX,offset HLTn2  // 62
  mov EAX,offset HLTn2
  mov EAX,offset Mix2   // 63
  mov EAX,offset RRA16
  mov EAX,offset RWz2   // 64
  mov EAX,offset NOP8
  mov EAX,offset RWz2   // 65
  mov EAX,offset ADC8
  mov EAX,offset Mz2    // 66
  mov EAX,offset ROR8
  mov EAX,offset Mz2    // 67
  mov EAX,offset RRA8
  mov EAX,offset PLx2   // 68
  mov EAX,offset PLAn4
  mov EAX,offset ADCb   // 69
  mov EAX,offset ADCb
  mov EAX,offset RORn   // 6A
  mov EAX,offset RORn
  mov EAX,offset ARRb   // 6B
  mov EAX,offset ARRb
  mov EAX,offset JMPi2  // 6C
  mov EAX,offset JMPi2
  mov EAX,offset RWa2   // 6D
  mov EAX,offset ADC16
  mov EAX,offset Ma2    // 6E
  mov EAX,offset ROR16
  mov EAX,offset Ma2    // 6F
  mov EAX,offset RRA16
  mov EAX,offset BVSr2  // 70
  mov EAX,offset BVSr2
  mov EAX,offset Riy2   // 71
  mov EAX,offset ADC16
  mov EAX,offset HLTn2  // 72
  mov EAX,offset HLTn2
  mov EAX,offset Miy2   // 73
  mov EAX,offset RRA16
  mov EAX,offset RWzx2  // 74
  mov EAX,offset NOP8
  mov EAX,offset RWzx2  // 75
  mov EAX,offset ADC8
  mov EAX,offset Mzx2   // 76
  mov EAX,offset ROR8
  mov EAX,offset Mzx2   // 77
  mov EAX,offset RRA8
  mov EAX,offset SEIn   // 78
  mov EAX,offset SEIn
  mov EAX,offset Ray2   // 79
  mov EAX,offset ADC16
  mov EAX,offset NOPn   // 7A
  mov EAX,offset NOPn
  mov EAX,offset May2   // 7B
  mov EAX,offset RRA16
  mov EAX,offset Rax2   // 7C
  mov EAX,offset NOP16
  mov EAX,offset Rax2   // 7D
  mov EAX,offset ADC16
  mov EAX,offset Max2   // 7E
  mov EAX,offset ROR16
  mov EAX,offset Max2   // 7F
  mov EAX,offset RRA16
  mov EAX,offset NOPb   // 80
  mov EAX,offset NOPb
  mov EAX,offset RWix2  // 81
  mov EAX,offset STA16
  mov EAX,offset NOPb   // 82
  mov EAX,offset NOPb
  mov EAX,offset RWix2  // 83
  mov EAX,offset AXS16
  mov EAX,offset RWz2   // 84
  mov EAX,offset STY8
  mov EAX,offset RWz2   // 85
  mov EAX,offset STA8
  mov EAX,offset RWz2   // 86
  mov EAX,offset STX8
  mov EAX,offset RWz2   // 87
  mov EAX,offset AXS8
  mov EAX,offset DEYn   // 88
  mov EAX,offset DEYn
  mov EAX,offset NOPb   // 89
  mov EAX,offset NOPb
  mov EAX,offset TXAn   // 8A
  mov EAX,offset TXAn
  mov EAX,offset ANEb   // 8B
  mov EAX,offset ANEb
  mov EAX,offset RWa2   // 8C
  mov EAX,offset STY16
  mov EAX,offset RWa2   // 8D
  mov EAX,offset STA16
  mov EAX,offset RWa2   // 8E
  mov EAX,offset STX16
  mov EAX,offset RWa2   // 8F
  mov EAX,offset AXS16
  mov EAX,offset BCCr2  // 90
  mov EAX,offset BCCr2
  mov EAX,offset Wiy2   // 91
  mov EAX,offset STA16
  mov EAX,offset HLTn2  // 92
  mov EAX,offset HLTn2
  mov EAX,offset Wiy2   // 93
  mov EAX,offset SHA16
  mov EAX,offset RWzx2  // 94
  mov EAX,offset STY8
  mov EAX,offset RWzx2  // 95
  mov EAX,offset STA8
  mov EAX,offset RWzy2  // 96
  mov EAX,offset STX8
  mov EAX,offset RWzy2  // 97
  mov EAX,offset AXS8
  mov EAX,offset TYAn   // 98
  mov EAX,offset TYAn
  mov EAX,offset Way2   // 99
  mov EAX,offset STA16
  mov EAX,offset TXSn   // 9A
  mov EAX,offset TXSn
  mov EAX,offset Way2   // 9B
  mov EAX,offset SHS16
  mov EAX,offset Wax2   // 9C
  mov EAX,offset SHY16
  mov EAX,offset Wax2   // 9D
  mov EAX,offset STA16
  mov EAX,offset Way2   // 9E
  mov EAX,offset SHX16
  mov EAX,offset Way2   // 9F
  mov EAX,offset SHA16
  mov EAX,offset LDYb   // A0
  mov EAX,offset LDYb
  mov EAX,offset RWix2  // A1
  mov EAX,offset LDA16
  mov EAX,offset LDXb   // A2
  mov EAX,offset LDXb
  mov EAX,offset RWix2  // A3
  mov EAX,offset LAX16
  mov EAX,offset RWz2   // A4
  mov EAX,offset LDY8
  mov EAX,offset RWz2   // A5
  mov EAX,offset LDA8
  mov EAX,offset RWz2   // A6
  mov EAX,offset LDX8
  mov EAX,offset RWz2   // A7
  mov EAX,offset LAX8
  mov EAX,offset TAYn   // A8
  mov EAX,offset TAYn
  mov EAX,offset LDAb   // A9
  mov EAX,offset LDAb
  mov EAX,offset TAXn   // AA
  mov EAX,offset TAXn
  mov EAX,offset LXAb   // AB
  mov EAX,offset LXAb
  mov EAX,offset RWa2   // AC
  mov EAX,offset LDY16
  mov EAX,offset RWa2   // AD
  mov EAX,offset LDA16
  mov EAX,offset RWa2   // AE
  mov EAX,offset LDX16
  mov EAX,offset RWa2   // AF
  mov EAX,offset LAX16
  mov EAX,offset BCSr2  // B0
  mov EAX,offset BCSr2
  mov EAX,offset Riy2   // B1
  mov EAX,offset LDA16
  mov EAX,offset HLTn2  // B2
  mov EAX,offset HLTn2
  mov EAX,offset Riy2   // B3
  mov EAX,offset LAX16
  mov EAX,offset RWzx2  // B4
  mov EAX,offset LDY8
  mov EAX,offset RWzx2  // B5
  mov EAX,offset LDA8
  mov EAX,offset RWzy2  // B6
  mov EAX,offset LDX8
  mov EAX,offset RWzy2  // B7
  mov EAX,offset LAX8
  mov EAX,offset CLVn   // B8
  mov EAX,offset CLVn
  mov EAX,offset Ray2   // B9
  mov EAX,offset LDA16
  mov EAX,offset TSXn   // BA
  mov EAX,offset TSXn
  mov EAX,offset Ray2   // BB
  mov EAX,offset LAS16
  mov EAX,offset Rax2   // BC
  mov EAX,offset LDY16
  mov EAX,offset Rax2   // BD
  mov EAX,offset LDA16
  mov EAX,offset Ray2   // BE
  mov EAX,offset LDX16
  mov EAX,offset Ray2   // BF
  mov EAX,offset LAX16
  mov EAX,offset CPYb   // C0
  mov EAX,offset CPYb
  mov EAX,offset RWix2  // C1
  mov EAX,offset CMP16
  mov EAX,offset NOPb   // C2
  mov EAX,offset NOPb
  mov EAX,offset Mix2   // C3
  mov EAX,offset DCM16
  mov EAX,offset RWz2   // C4
  mov EAX,offset CPY8
  mov EAX,offset RWz2   // C5
  mov EAX,offset CMP8
  mov EAX,offset Mz2    // C6
  mov EAX,offset DEC8
  mov EAX,offset Mz2    // C7
  mov EAX,offset DCM8
  mov EAX,offset INYn   // C8
  mov EAX,offset INYn
  mov EAX,offset CMPb   // C9
  mov EAX,offset CMPb
  mov EAX,offset DEXn   // CA
  mov EAX,offset DEXn
  mov EAX,offset SBXb   // CB
  mov EAX,offset SBXb
  mov EAX,offset RWa2   // CC
  mov EAX,offset CPY16
  mov EAX,offset RWa2   // CD
  mov EAX,offset CMP16
  mov EAX,offset Ma2    // CE
  mov EAX,offset DEC16
  mov EAX,offset Ma2    // CF
  mov EAX,offset DCM16
  mov EAX,offset BNEr2  // D0
  mov EAX,offset BNEr2
  mov EAX,offset Riy2   // D1
  mov EAX,offset CMP16
  mov EAX,offset HLTn2  // D2
  mov EAX,offset HLTn2
  mov EAX,offset Miy2   // D3
  mov EAX,offset DCM16
  mov EAX,offset RWzx2  // D4
  mov EAX,offset NOP8
  mov EAX,offset RWzx2  // D5
  mov EAX,offset CMP8
  mov EAX,offset Mzx2   // D6
  mov EAX,offset DEC8
  mov EAX,offset Mzx2   // D7
  mov EAX,offset DCM8
  mov EAX,offset CLDn   // D8
  mov EAX,offset CLDn
  mov EAX,offset Ray2   // D9
  mov EAX,offset CMP16
  mov EAX,offset NOPn   // DA
  mov EAX,offset NOPn
  mov EAX,offset May2   // DB
  mov EAX,offset DCM16
  mov EAX,offset Rax2   // DC
  mov EAX,offset NOP16
  mov EAX,offset Rax2   // DD
  mov EAX,offset CMP16
  mov EAX,offset Max2   // DE
  mov EAX,offset DEC16
  mov EAX,offset Max2   // DF
  mov EAX,offset DCM16
  mov EAX,offset CPXb   // E0
  mov EAX,offset CPXb
  mov EAX,offset RWix2  // E1
  mov EAX,offset SBC16
  mov EAX,offset NOPb   // E2
  mov EAX,offset NOPb
  mov EAX,offset Mix2   // E3
  mov EAX,offset INS16
  mov EAX,offset RWz2   // E4
  mov EAX,offset CPX8
  mov EAX,offset RWz2   // E5
  mov EAX,offset SBC8
  mov EAX,offset Mz2    // E6
  mov EAX,offset INC8
  mov EAX,offset Mz2    // E7
  mov EAX,offset INS8
  mov EAX,offset INXn   // E8
  mov EAX,offset INXn
  mov EAX,offset SBCb   // E9
  mov EAX,offset SBCb
  mov EAX,offset NOPn   // EA
  mov EAX,offset NOPn
  mov EAX,offset SBCb   // EB
  mov EAX,offset SBCb
  mov EAX,offset RWa2   // EC
  mov EAX,offset CPX16
  mov EAX,offset RWa2   // ED
  mov EAX,offset SBC16
  mov EAX,offset Ma2    // EE
  mov EAX,offset INC16
  mov EAX,offset Ma2    // EF
  mov EAX,offset INS16
  mov EAX,offset BEQr2  // F0
  mov EAX,offset BEQr2
  mov EAX,offset Riy2   // F1
  mov EAX,offset SBC16
  mov EAX,offset HLTn2  // F2
  mov EAX,offset HLTn2
  mov EAX,offset Miy2   // F3
  mov EAX,offset INS16
  mov EAX,offset RWzx2  // F4
  mov EAX,offset NOP8
  mov EAX,offset RWzx2  // F5
  mov EAX,offset SBC8
  mov EAX,offset Mzx2   // F6
  mov EAX,offset INC8
  mov EAX,offset Mzx2   // F7
  mov EAX,offset INS8
  mov EAX,offset SEDn   // F8
  mov EAX,offset SEDn
  mov EAX,offset Ray2   // F9
  mov EAX,offset SBC16
  mov EAX,offset NOPn   // FA
  mov EAX,offset NOPn
  mov EAX,offset May2   // FB
  mov EAX,offset INS16
  mov EAX,offset Rax2   // FC
  mov EAX,offset NOP16
  mov EAX,offset Rax2   // FD
  mov EAX,offset SBC16
  mov EAX,offset Max2   // FE
  mov EAX,offset INC16
  mov EAX,offset Max2   // FF
  mov EAX,offset INS16


///////////////////////////////////////////////////////////////////////////////
// When the CPU65xx object loads itself from the disk, the indices must be
// converted back to function pointers. Another address table is needed. To
// ensure that this table is correct, it will be checked within OnLoad.

// TODO: add table
  mov EAX,0 // end of table for check

  mov EAX,offset Reset1
  mov EAX,offset Reset2
  mov EAX,offset IRQ1
  mov EAX,offset IRQ2
  mov EAX,offset IRQ3
  mov EAX,offset IRQ4
  mov EAX,offset IRQ5
  mov EAX,offset IRQ6
  mov EAX,offset IRQ7
  mov EAX,offset NMI1
  mov EAX,offset NMI2
  mov EAX,offset NMI3
  mov EAX,offset NMI4
  mov EAX,offset NMI5
  mov EAX,offset NMI6
  mov EAX,offset NMI7
  mov EAX,offset RDY
  mov EAX,offset Command


////////////////////////////////////////////////////////////////////////////////
// Reset  TODO: must be clocks
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1   FFFC   R  fetch address ll                                Reset1
//  2   FFFD   R  fetch address hh, PC = hhll                     Reset2

proc(Reset1) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[(0xFFFC shr 12)*4]
  mov [ESI]Chip.pfnOnClock,offset Reset2+4

  xor EDX,EDX
  mov EBP,[ESI]Chip.pNextChip

  mov DL,[EBX+0xFFFC+65536]
  mov AL,[EBX+0xFFFC]

  cmp DL,CPU16x4k_iIOStart
  jae Reset1_Trap

Reset1_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

Reset1_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Reset1_Continue

endp

proc(Reset2) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[(0xFFFD shr 12)*4]
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EAX,[ESI]CPU65xx.iAddr16
  mov [ESI]Chip.pfnOnClock,EDX

  mov AH,[EBX+0xFFFD]
  mov EBP,[ESI]Chip.pNextChip

  mov DL,[EBX+0xFFFD+65536]
  ;nop

  cmp DL,CPU16x4k_iIOStart
  jae Reset2_Trap

Reset2_Continue:

  mov EBX,EAX
  ;nop

  shr EBX,12
  ;nop

  ;AGI
  ;stall

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EBX*4]
  ;nop

  mov [ESI]CPU65xx.pbPCBase,EBX
  add EAX,EBX

  mov [ESI]CPU65xx.pbPCPtr,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

Reset2_Trap:

  and EDX,0x000000FF
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov BL,AL
  mov EAX,[ESI]CPU65xx.iAddr16
  mov AH,BL
  jmp Reset2_Continue

endp


////////////////////////////////////////////////////////////////////////////////
// IRQ - Interrupt Request
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  dummy fetch                                     IRQ1
//  2    PC    R  dummy fetch                                     IRQ2
//  3   01SP   W  push PCH, SP -= 1                               IRQ3
//  4   01SP   W  push PCL, SP -= 1                               IRQ4
//  5   01SP   W  push flags with BF cleared, SP -= 1, set IF     IRQ5
//  6   FFFE   R  read ll                                         IRQ6
//  7   FFFF   R  read hh, PC = hhll                              IRQ7

JmpCommandPlus4: // TODO: remove
  jmp Command+4

proc(IRQ1) index(0)

  mov ECX,[ESI]Chip.pClock
  mov EBP,[ESI]CPU65xx.iPostponeIRQ

  mov EAX,[ESI]CPU65xx.iInterrupts
  mov EBX,[ESI]CPU65xx.pbPCPtr

  mov EDI,[ECX]Clock.TimerRoot.pNextTimer
  mov EDX,[ECX]Clock.TimerRoot.iClocks

  and EAX,~giIRQ
  sub EBP,EDX

  mov EDX,offset IRQ2+4
  mov ECX,[EDI]Timer.iClocks

  cmp EBP,ECX
  jns JmpCommandPlus4

  mov [ESI]CPU65xx.iInterrupts,EAX
  mov [ESI]Chip.pfnOnClock,EDX

  mov EAX,gapfnInterrupt[EAX*4]
  mov EBP,ESI

  mov [ESI]CPU65xx.pfnInterrupt,EAX
  mov ESI,[ESI]Chip.pNextChip

  mov DL,[EBX+65536]
  ;nop

  test DL,DL
  jne IRQ1_Special

IRQ1_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(IRQ2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset IRQ3+4

  mov EBP,ESI
  mov [ESI]Chip.pfnOnClock,EDX

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+65536]

  test DL,DL
  jne IRQ2_Special

IRQ2_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(IRQ3) index(0)

  mov EDX,offset IRQ4+4
  mov ECX,[ESI]CPU65xx.iSP

  mov [ESI]Chip.pfnOnClock,EDX
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov EBP,ESI
  lea EAX,[ECX-1]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iSP),AL

  mov EAX,[ESI]CPU65xx.pbPCPtr
  mov EDI,[ESI]CPU65xx.pbPCBase
  
  sub EAX,EDI
  mov ESI,[ESI]Chip.pNextChip

  mov [EBX+ECX],AH
  mov AL,AH

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(IRQ4) index(0)

  mov EDX,offset IRQ5+4
  mov ECX,[ESI]CPU65xx.iSP

  mov [ESI]Chip.pfnOnClock,EDX
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov EBP,ESI  
  lea EAX,[ECX-1]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iSP),AL

  mov EAX,[ESI]CPU65xx.pbPCPtr
  mov EDI,[ESI]CPU65xx.pbPCBase

  sub EAX,EDI
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(IRQ5) index(0)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,offset IRQ6+4
  mov EBP,ESI

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX-1]

  mov byte ptr ([ESI]CPU65xx.iSP),AL
  mov EAX,[ESI]CPU65xx.iNCFlag

  mov DL,[ESI]CPU65xx.bVFlag
  and EAX,0000000110000000b

  shl DL,6
  or AL,AH

  mov DH,[ESI]CPU65xx.bZVal
  or AL,DL

  mov DL,[ESI]CPU65xx.bFlags
  cmp DH,0

  jne IRQ5_NoZero
  or AL,00000010b
IRQ5_NoZero:

  or AL,DL
  or DL,0x04

  and AL,11101111b
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.bFlags,DL
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(IRQ6) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[(0xFFFE shr 12)*4]
  mov EDX,offset IRQ7+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+0xFFFE+65536]
  mov AL,[EBX+0xFFFE]

  cmp DL,CPU16x4k_iIOStart
  jae IRQ6_Trap

IRQ6_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(IRQ7) index(0)

  mov EDX,offset Command+4
  mov EBX,[ESI]CPU16x4k.apbReadMapping[(0xFFFF shr 12)*4]

  mov [ESI]Chip.pfnOnClock,EDX
  nop

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EBP,[ESI]Chip.pNextChip

  mov DL,[EBX+0xFFFF+65536]
  mov AH,[EBX+0xFFFF]

  cmp DL,CPU16x4k_iIOStart
  jae IRQ7_Trap

IRQ7_Continue:

  mov EBX,EAX
  shr EBX,12
  mov EBX,[ESI]CPU16x4k.apbReadMapping[EBX*4]
  mov [ESI]CPU65xx.pbPCBase,EBX
  add EAX,EBX
  mov [ESI]CPU65xx.pbPCPtr,EAX
  mov ESI,EBP
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// NMI - Interrupt Request
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  dummy fetch                                     NMI1
//  2    PC    R  dummy fetch                                     NMI2
//  3   01SP   W  push PCH, SP -= 1                               NMI3
//  4   01SP   W  push PCL, SP -= 1                               NMI4
//  5   01SP   W  push flags with BF cleared, SP -= 1, set IF     NMI5
//  6   FFFA   R  read ll                                         NMI6
//  7   FFFB   R  read hh, PC = hhll                              NMI7

proc(NMI1) index(0)

  mov ECX,[ESI]Chip.pClock
  mov EBP,[ESI]CPU65xx.iPostponeNMI

  mov EAX,[ESI]CPU65xx.iInterrupts
  mov EBX,[ESI]CPU65xx.pbPCPtr

  mov EDI,[ECX]Clock.TimerRoot.pNextTimer
  mov EDX,[ECX]Clock.TimerRoot.iClocks

  and EAX,~giNMI
  sub EBP,EDX

  mov EDX,offset NMI2+4
  mov ECX,[EDI]Timer.iClocks

  cmp EBP,ECX
  jns JmpCommandPlus4

  mov [ESI]CPU65xx.iInterrupts,EAX
  mov [ESI]Chip.pfnOnClock,EDX

  mov EAX,gapfnInterrupt[EAX*4]
  mov EBP,ESI

  mov [ESI]CPU65xx.pfnInterrupt,EAX
  mov ESI,[ESI]Chip.pNextChip

  mov DL,[EBX+65536]
  ;nop

  test DL,DL
  jne NMI1_Special

NMI1_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(NMI2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset NMI3+4

  mov EBP,ESI
  mov [ESI]Chip.pfnOnClock,EDX

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+65536]

  test DL,DL
  jne NMI2_Special

NMI2_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(NMI3) index(0)

  mov EDX,offset NMI4+4
  mov ECX,[ESI]CPU65xx.iSP

  mov [ESI]Chip.pfnOnClock,EDX
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov EBP,ESI
  lea EAX,[ECX-1]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iSP),AL

  mov EAX,[ESI]CPU65xx.pbPCPtr
  mov EDI,[ESI]CPU65xx.pbPCBase
  
  sub EAX,EDI
  mov ESI,[ESI]Chip.pNextChip

  mov [EBX+ECX],AH
  mov AL,AH

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(NMI4) index(0)

  mov EDX,offset NMI5+4
  mov ECX,[ESI]CPU65xx.iSP

  mov [ESI]Chip.pfnOnClock,EDX
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov EBP,ESI
  lea EAX,[ECX-1]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iSP),AL

  mov EAX,[ESI]CPU65xx.pbPCPtr
  mov EDI,[ESI]CPU65xx.pbPCBase

  sub EAX,EDI
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(NMI5) index(0)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,offset NMI6+4
  mov EBP,ESI

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX-1]

  mov byte ptr ([ESI]CPU65xx.iSP),AL
  mov EAX,[ESI]CPU65xx.iNCFlag

  mov DL,[ESI]CPU65xx.bVFlag
  and EAX,0000000110000000b

  shl DL,6
  or AL,AH

  mov DH,[ESI]CPU65xx.bZVal
  or AL,DL

  mov DL,[ESI]CPU65xx.bFlags
  cmp DH,0

  jne NMI5_NoZero
  or AL,00000010b
NMI5_NoZero:

  or AL,DL
  or DL,0x04

  and AL,11101111b
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.bFlags,DL
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(NMI6) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[(0xFFFA shr 12)*4]
  mov EDX,offset NMI7+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+0xFFFA+65536]
  mov AL,[EBX+0xFFFA]

  ;cache bank
  ;conflict

  cmp DL,CPU16x4k_iIOStart
  jae NMI6_Trap

NMI6_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(NMI7) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[(0xFFFB shr 12)*4]
  mov EDX,offset Command+4

  mov EAX,[ESI]CPU65xx.iAddr16
  mov [ESI]Chip.pfnOnClock,EDX

  mov AH,[EBX+0xFFFB]
  mov EBP,[ESI]Chip.pNextChip

  mov DL,[EBX+0xFFFB+65536]
  mov EBX,EAX

  cmp DL,CPU16x4k_iIOStart
  jae NMI7_Trap

NMI7_Continue:

  shr EBX,12
  ;nop

  ;AGI
  ;stall

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EBX*4]
  ;nop

  mov [ESI]CPU65xx.pbPCBase,EBX
  add EAX,EBX

  mov [ESI]CPU65xx.pbPCPtr,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


///////////////////////////////////////////////////////////////////////////////
// RDY - writing command has finished, halt the CPU now

proc(RDY) index(0)

  // halt the CPU
  mov ECX,ESI
  call CPU_SetIdle

  // clear RDY interrupt
  mov EAX,[ESI]CPU65xx.iInterrupts
  and EAX,~giRDY
  mov [ESI]CPU65xx.iInterrupts,EAX
  mov EAX,gapfnInterrupt[EAX*4]
  mov [ESI]CPU65xx.pfnInterrupt,EAX

  // set function when CPU is restarted again
  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EAX
  mov ESI,EBP
  jmp [EBP]Chip.pfnOnClock

endp


///////////////////////////////////////////////////////////////////////////////
// write to IO register
//
// EBP = this
// DL  = trap
// ECX = C64 address
// AL  = data
// ESI = pNextChip

WriteReg:

  mov EBX,[ESI]Chip.pfnOnClock
  and EDX,0x000000FF

  push EAX
  push EBX

  mov [EBP]CPU65xx.iRWAddress,ECX
  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]CPU16x4k.apfnbOnWrite[EDX*4]


///////////////////////////////////////////////////////////////////////////////
// Command - Fetch the next byte and jump to command table
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command

proc(Command) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EBP,ESI

  xor EAX,EAX
  mov ESI,[ESI]Chip.pNextChip

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [EBP]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Command_Special

Command_Continue:

  mov ECX,gapfnCommand[EAX*8+0]
  mov EAX,gapfnCommand[EAX*8+4]

  mov [EBP]Chip.pfnOnClock,ECX
  mov [EBP]CPU65xx.pfnAction,EAX

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// RWz - Read/write command, zeropage addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       RWz2
//  3   00ll   R  read data, perform action                       pfnAction
// or:
//  3   00ll   W  perform action, write data                      pfnAction

proc(RWz2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnAction

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne RWz2_Special

RWz2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// RWzx - Read/write command, zeropage+x addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       RWzx2
//  3   00ll   R  dummy read, ll += xreg                          RWzx3
//  4   00ll   R  read data, perform action                       pfnAction
// or:
//  4   00ll   W  perform action, write data                      pfnAction

proc(RWzx2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset RWzx3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne RWzx2_Special

RWzx2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(RWzx3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnAction
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)

  mov [ESI]Chip.pfnOnClock,EDX
  add AL,CL

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iDummyStart
  jae RWzx3_Trap

RWzx3_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// RWzy - Read/write command, zeropage+y addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       RWzy2
//  3   00ll   R  dummy read, ll += yreg                          RWzy3
//  4   00ll   R  read data, perform action                       pfnAction
// or:
//  4   00ll   W  perform action, write data                      pfnAction

proc(RWzy2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset RWzy3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne RWzy2_Special

RWzy2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(RWzy3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnAction
  mov AL,byte ptr ([ESI]CPU65xx.iYReg)

  mov [ESI]Chip.pfnOnClock,EDX
  add AL,CL

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iDummyStart
  jae RWzy3_Trap

RWzy3_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// RWa - Read/write command, absolute addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       RWa2
//  3    PC    R  fetch address hh, PC += 1                       RWa3
//  4   hhll   R  read data, perform action                       pfnAction
// or:
//  4   hhll   W  perform action, write data                      pfnAction

proc(RWa2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset RWa3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne RWa2_Special

RWa2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(RWa3) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnAction

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne RWa3_Special

RWa3_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Rax - Read command, absolute+x addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       Rax2
//  3    PC    R  fetch address hh, PC += 1     , ll += yreg      Rax3
//  4   hhll   R  read data, perform action                       pfnAction
// if carry(ll):
//  4   hhll   R  dummy read, hh += 1                             Raxy4
//  5   hhll   R  read data, perform action                       pfnAction

proc(Rax2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Rax3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Rax2_Special

Rax2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Rax3) index(0)

  mov EAX,[ESI]CPU65xx.iXReg
  mov CL,byte ptr ([ESI]CPU65xx.iAddr16[0])

  mov EBX,[ESI]CPU65xx.pbPCPtr
  add AL,CL

  mov EDX,[ESI]CPU65xx.pfnAction
  jnc Rax3_NoFixAdr

  mov EDX,offset Raxy4+4
  nop

Rax3_NoFixAdr:

  mov [ESI]Chip.pfnOnClock,EDX
  mov EBP,[ESI]Chip.pNextChip

  mov AH,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Rax3_Special

Rax3_Continue:

  mov [ESI]CPU65xx.iAddr16,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Raxy4) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnAction

  shr EAX,12
  mov ECX,[ESI]CPU65xx.iAddr16

  mov [ESI]Chip.pfnOnClock,EDX
  mov DL,CH

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  inc DL

  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),DL
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iDummyStart
  jae Raxy4_Trap

Raxy4_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Ray - Read command, absolute+y addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       Ray2
//  3    PC    R  fetch address hh, PC += 1, ll += yreg           Ray3
//  4   hhll   R  read data, perform action                       pfnAction
// if carry(ll):
//  4   hhll   R  dummy read, hh += 1                             Raxy4
//  5   hhll   R  read data, perform action                       pfnAction

proc(Ray2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Ray3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Ray2_Special

Ray2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Ray3) index(0)

  mov EAX,[ESI]CPU65xx.iYReg
  mov CL,byte ptr ([ESI]CPU65xx.iAddr16[0])

  mov EBX,[ESI]CPU65xx.pbPCPtr
  add AL,CL

  mov EDX,[ESI]CPU65xx.pfnAction
  jnc Ray3_NoFixAdr

  mov EDX,offset Raxy4+4
  ;nop

Ray3_NoFixAdr:

  mov [ESI]Chip.pfnOnClock,EDX
  mov EBP,[ESI]Chip.pNextChip

  mov AH,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Ray3_Special

Ray3_Continue:

  mov [ESI]CPU65xx.iAddr16,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Wax - Write command, absolute+x addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       Wax2
//  3    PC    R  fetch address hh, PC += 1, ll += xreg           Wax3
//  4   hhll   R  dummy read, hh += carry(ll)                     Waxy4
//  5   hhll   W  perform action, write data                      pfnAction

proc(Wax2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Wax3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Wax2_Special

Wax2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Wax3) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  xor ECX,ECX

  mov EAX,[ESI]CPU65xx.iXReg
  mov CL,byte ptr ([ESI]CPU65xx.iAddr16[0])

  add EAX,ECX
  mov EDX,offset Waxy4+4

  lea ECX,[EBX+1]
  mov [ESI]Chip.pfnOnClock,EDX

  mov [ESI]CPU65xx.pbPCPtr,ECX

  xor EDX,EDX

  mov EBP,[ESI]Chip.pNextChip
  mov DL,[EBX+65536]

  mov [ESI]CPU65xx.bFixAdr,AH
  mov AH,[EBX]

  test DL,DL
  jne Wax3_Special

Wax3_Continue:

  mov [ESI]CPU65xx.iAddr16,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Waxy4) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnAction

  shr EAX,12
  mov ECX,[ESI]CPU65xx.iAddr16

  mov [ESI]Chip.pfnOnClock,EDX
  mov DL,[ESI]CPU65xx.bFixAdr

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add DL,CH

  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),DL
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iDummyStart
  jae Waxy4_Trap

Waxy4_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Way - Write command, absolute+y addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       Way2
//  3    PC    R  fetch address hh, PC += 1, ll += yreg           Way3
//  4   hhll   R  dummy read, hh += carry(ll)                     Waxy4
//  5   hhll   W  perform action, write data                      pfnAction

proc(Way2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Way3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Way2_Special

Way2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Way3) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  xor ECX,ECX

  mov EAX,[ESI]CPU65xx.iYReg
  mov CL,byte ptr ([ESI]CPU65xx.iAddr16[0])

  add EAX,ECX
  mov EDX,offset Waxy4+4

  lea ECX,[EBX+1]
  mov [ESI]Chip.pfnOnClock,EDX

  mov [ESI]CPU65xx.pbPCPtr,ECX
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bFixAdr,AH
  xor EDX,EDX

  mov AH,[EBX]
  mov DL,[EBX+65536]

  test DL,DL
  jne Way3_Special

Way3_Continue:

  mov [ESI]CPU65xx.iAddr16,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// RWix - Read/write command, indirect+x addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch pointer pp, PC += 1                       RWix2
//  3   00pp   R  dummy read, pp += xreg                          RWix3
//  4   00pp   R  read address ll, pp += 1                        RWix4
//  5   00pp   R  read address hh                                 RWix5
//  6   hhll   R  read data, perform action                       pfnAction
// or:
//  6   hhll   W  perform action, write data                      pfnAction

proc(RWix2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset RWix3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne RWix2_Special

RWix2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(RWix3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,offset RWix4+4
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)

  mov [ESI]Chip.pfnOnClock,EDX
  add AL,CL

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iDummyStart
  jae RWix3_Trap

RWix3_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(RWix4) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  xor EDX,EDX
  mov EBP,offset RWix5+4

  mov DL,[EBX+ECX+65536]
  lea EAX,[ECX+1]

  mov [ESI]Chip.pfnOnClock,EBP
  mov byte ptr ([ESI]CPU65xx.iAddr8),AL

  mov EBP,[ESI]Chip.pNextChip
  mov AL,[EBX+ECX]

  cmp DL,CPU16x4k_iIOStart
  jae RWix4_Trap

RWix4_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(RWix5) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  xor EDX,EDX
  mov EBP,[ESI]CPU65xx.pfnAction

  mov AL,[EBX+ECX]
  mov [ESI]Chip.pfnOnClock,EBP

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae RWix5_Trap

RWix5_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Riy - Read command, indirect+y addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch pointer pp, PC += 1                       Riy2
//  3   00pp   R  read address ll, pp += 1                        Riy3
//  4   00pp   R  read address hh, ll += yreg                     Riy4
//  5   hhll   R  read data, perform action                       pfnAction
// if carry(ll):
//  5   hhll   R  dummy read, hh += 1                             Raxy4
//  6   hhll   R  read data, perform action                       pfnAction

proc(Riy2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Riy3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Riy2_Special

Riy2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Riy3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EBP,offset Riy4+4
  xor EDX,EDX

  mov [ESI]Chip.pfnOnClock,EBP
  lea EAX,[ECX+1]

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae Riy3_Trap

Riy3_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Riy4) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov ECX,[ESI]CPU65xx.iYReg

  add AL,CL
  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnAction
  jnc Riy4_NoFixAdr

  mov EDX,offset Raxy4+4
  nop

Riy4_NoFixAdr:

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae Riy4_Trap

Riy4_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Wiy - Write command, indirect+y addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch pointer pp, PC += 1                       Wiy2
//  3   00pp   R  read address ll, pp += 1                        Wiy3
//  4   00pp   R  read address hh, ll += yreg                     Wiy4
//  5   hhll   R  dummy read, hh += carry(ll)                     Waxy4
//  6   hhll   R  perform action, write data                      pfnAction

proc(Wiy2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Wiy3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Wiy2_Special

Wiy2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Wiy3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EBP,offset Wiy4+4
  xor EDX,EDX

  mov [ESI]Chip.pfnOnClock,EBP
  lea EAX,[ECX+1]

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae Wiy3_Trap

Wiy3_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Wiy4) index(0)

  xor EAX,EAX
  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]

  mov AL,byte ptr ([ESI]CPU65xx.iAddr16)
  mov EDX,[ESI]CPU65xx.iYReg

  add EAX,EDX
  mov ECX,[ESI]CPU65xx.iAddr8

  mov [ESI]CPU65xx.bFixAdr,AH
  mov EBP,offset Waxy4+4

  mov AH,[EBX+ECX]
  mov [ESI]Chip.pfnOnClock,EBP

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae Wiy4_Trap

Wiy4_Continue:

  mov [ESI]CPU65xx.iAddr16,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Mz - Modify command, zeropage addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       Mz2
//  3   00ll   R  read data                                       Mz3
//  4   00ll   W  write old data, perform action                  pfnAction
//  5   00ll   W  write new data                                  Mz5

proc(Mz2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Mz3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Mz2_Special

Mz2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Mz3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  xor EDX,EDX
  mov EBP,[ESI]CPU65xx.pfnAction

  mov AL,[EBX+ECX]
  mov [ESI]Chip.pfnOnClock,EBP

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae Mz3_Trap

Mz3_Continue:

  mov [ESI]CPU65xx.bData,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Mz5) index(0)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov AL,[ESI]CPU65xx.bData

  mov [ESI]Chip.pfnOnClock,EDX
  mov EBP,ESI

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Mzx - Modify command, zeropage+x addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       Mzx2
//  3   00ll   R  dummy read, ll += x reg                         Mzx3
//  4   00ll   R  read data                                       Mzx4
//  5   00ll   W  write old data, perform action                  pfnAction
//  6   00ll   W  write new data                                  Mz5

proc(Mzx2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Mzx3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Mzx2_Special

Mzx2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Mzx3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,offset Mzx4+4
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)

  mov [ESI]Chip.pfnOnClock,EDX
  add AL,CL

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iDummyStart
  jae Mzx3_Trap

Mzx3_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Mzx4) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  xor EDX,EDX
  mov EBP,[ESI]CPU65xx.pfnAction

  mov AL,[EBX+ECX]
  mov [ESI]Chip.pfnOnClock,EBP

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae Mzx4_Trap

Mzx4_Continue:

  mov [ESI]CPU65xx.bData,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Ma - Modify command, absolute addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       Ma2
//  3    PC    R  fetch address hh, PC += 1                       Ma3
//  4   hhll   R  read data                                       Ma4
//  5   hhll   W  write old data, perform action                  pfnAction
//  6   hhll   W  write new data                                  Ma6

proc(Ma2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Ma3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Ma2_Special

Ma2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Ma3) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Ma4+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Ma3_Special

Ma3_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Ma4) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnAction

  shr EAX,12
  nop

  mov ECX,[ESI]CPU65xx.iAddr16
  mov [ESI]Chip.pfnOnClock,EDX

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  nop

  mov EBP,[ESI]Chip.pNextChip
  xor EDX,EDX

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae Ma4_Trap

Ma4_Continue:

  mov [ESI]CPU65xx.bData,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Ma6) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  mov [ESI]Chip.pfnOnClock,EDX

  mov EBP,ESI
  nop

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EAX*4]
  mov ECX,[ESI]CPU65xx.iAddr16

  mov AL,[ESI]CPU65xx.bData
  mov ESI,[ESI]Chip.pNextChip

  mov DL,[EBX+ECX+65536]
  mov [EBX+ECX],AL

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Max - Modify command, absolute+x addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       Max2
//  3    PC    R  fetch address hh, PC += 1, ll += xreg           Max3
//  4   hhll   R  dummy read, hh += carry(ll)                     Maxy4
//  5   hhll   R  read data                                       Ma4
//  6   hhll   W  write old data, perform action                  pfnAction
//  7   hhll   W  write new data                                  Ma6

proc(Max2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Max3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Max2_Special

Max2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Max3) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  xor ECX,ECX

  mov EAX,[ESI]CPU65xx.iXReg
  mov CL,byte ptr ([ESI]CPU65xx.iAddr16[0])

  add EAX,ECX
  mov EDX,offset Maxy4+4

  lea ECX,[EBX+1]
  mov [ESI]Chip.pfnOnClock,EDX

  mov [ESI]CPU65xx.pbPCPtr,ECX
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bFixAdr,AH
  xor EDX,EDX

  mov AH,[EBX]
  mov DL,[EBX+65536]

  test DL,DL
  jne Max3_Special

Max3_Continue:

  mov [ESI]CPU65xx.iAddr16,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Maxy4) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,offset Ma4+4

  shr EAX,12
  mov ECX,[ESI]CPU65xx.iAddr16

  mov [ESI]Chip.pfnOnClock,EDX
  mov DL,[ESI]CPU65xx.bFixAdr

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add DL,CH

  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),DL
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iDummyStart
  jae Maxy4_Trap

Maxy4_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// BEQr - Branch on equal (no flags), relative addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  2    PC    R  fetch offset xx, PC += 1, check for condition   BEQr2

proc(BEQr2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EBP,[ESI]Chip.pNextChip

  nop
  mov AH,[ESI]CPU65xx.bZVal

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne BEQr2_Special

BEQr2_Continue:

  cmp AH,0
  jne BEQr2_False

  mov EDX,offset Br3+4
  mov byte ptr ([ESI]CPU65xx.iAddr8),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

BEQr2_False:

  nop
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// BNEr - Branch on not equal (no flags), relative addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  2    PC    R  fetch offset xx, PC += 1, check for condition   BNEr2

proc(BNEr2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EBP,[ESI]Chip.pNextChip

  nop
  mov AH,[ESI]CPU65xx.bZVal

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne BNEr2_Special

BNEr2_Continue:

  cmp AH,0
  je BNEr2False

  mov EDX,offset Br3+4
  mov byte ptr ([ESI]CPU65xx.iAddr8),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

BNEr2False:

  nop
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// BMIr - Branch on minus (no flags), relative addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  2    PC    R  fetch offset xx, PC += 1, check for condition   BMIr2

proc(BMIr2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EBP,[ESI]Chip.pNextChip

  nop
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[0])

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne BMIr2_Special

BMIr2_Continue:

  test EAX,0x8000
  je BMIr2_False

  mov EDX,offset Br3+4
  mov byte ptr ([ESI]CPU65xx.iAddr8),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

BMIr2_False:

  nop
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// BPLr - Branch on plus (no flags), relative addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  2    PC    R  fetch offset xx, PC += 1, check for condition   BPLr2

proc(BPLr2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EBP,[ESI]Chip.pNextChip

  nop
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[0])

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne BPLr2_Special

BPLr2_Continue:

  test EAX,0x8000
  jne BPLr2_False

  mov EDX,offset Br3+4
  mov byte ptr ([ESI]CPU65xx.iAddr8),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

BPLr2_False:

  nop
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// BCSr - Branch on carry set (no flags), relative addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  2    PC    R  fetch offset xx, PC += 1, check for condition   BCSr2

proc(BCSr2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EBP,[ESI]Chip.pNextChip

  nop
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne BCSr2_Special

BCSr2_Continue:

  test EAX,0x0100
  je BCSr2_False

  mov EDX,offset Br3+4
  mov byte ptr ([ESI]CPU65xx.iAddr8),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

BCSr2_False:

  nop
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// BCCr - Branch on carry clear (no flags), relative addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  2    PC    R  fetch offset xx, PC += 1, check for condition   BCCr2

proc(BCCr2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EBP,[ESI]Chip.pNextChip

  nop
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne BCCr2_Special

BCCr2_Continue:

  test EAX,0x0100
  jne BCCr2_False

  mov EDX,offset Br3+4
  mov byte ptr ([ESI]CPU65xx.iAddr8),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

BCCr2_False:

  nop
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// BVSr - Branch on oVerflow set (no flags), relative addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  2    PC    R  fetch offset xx, PC += 1, check for condition   BVSr2

proc(BVSr2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EBP,[ESI]Chip.pNextChip

  nop
  mov AH,[ESI]CPU65xx.bVFlag

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne BVSr2_Special

BVSr2_Continue:

  cmp AH,0
  je BVSr2_False

  mov EDX,offset Br3+4
  mov byte ptr ([ESI]CPU65xx.iAddr8),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

BVSr2_False:

  nop
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// BVCr - Branch on oVerflow set (no flags), relative addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  2    PC    R  fetch offset xx, PC += 1, check for condition   BVCr2

proc(BVCr2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EBP,[ESI]Chip.pNextChip

  nop
  mov AH,[ESI]CPU65xx.bVFlag

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne BVCr2_Special

BVCr2_Continue:

  cmp AH,0
  jne BVCr2_False

  mov EDX,offset Br3+4
  mov byte ptr ([ESI]CPU65xx.iAddr8),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

BVCr2_False:

  nop
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Br - Branch (no flags), relative addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch offset xx, PC += 1, check for condition   pfnAction
// if condition is true:
//  3    PC    R  PCL += xx                                       Br3
// if condition is true and carry(PCL):
//  4    PC    R  PCH += 1                                        Br4

proc(Br3) index(0)

  mov DL,7Fh
  mov EAX,[ESI]CPU65xx.iAddr8

  cmp DL,AL
  mov EBX,[ESI]CPU65xx.pbPCPtr

  sbb AH,AH
  mov ECX,[ESI]CPU65xx.pbPCBase

  mov DL,[EBX+65536]
  sub EBX,ECX

  test DL,DL
  jne Br3_Special

Br3_Continue:

  add EAX,EBX
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  and EAX,0x0000FFFF
  mov EBP,[ESI]Chip.pNextChip

  cmp AH,BH
  je Br3_SamePage

  mov [ESI]CPU65xx.iAddr16,EAX
  mov EDX,offset Br4+4

  mov AH,BH
  jmp Br3_DifferentPage

Br3_SamePage:

  mov EBX,[ESI]CPU65xx.iPostponeIRQ
  mov EDI,[ESI]CPU65xx.iPostponeNMI

  inc EBX
  inc EDI

  mov [ESI]CPU65xx.iPostponeIRQ,EBX
  mov [ESI]CPU65xx.iPostponeNMI,EDI

Br3_DifferentPage:

  add EAX,ECX
  mov [ESI]Chip.pfnOnClock,EDX

  mov [ESI]CPU65xx.pbPCPtr,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Br4) index(0)

  mov ECX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr ECX,12
  mov [ESI]Chip.pfnOnClock,EDX

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EBP,[ESI]Chip.pNextChip

  mov ECX,[ESI]CPU16x4k.apbReadMapping[ECX*4]
  mov EBX,[ESI]CPU65xx.pbPCPtr

  add EAX,ECX
  mov [ESI]CPU65xx.pbPCBase,ECX

  mov [ESI]CPU65xx.pbPCPtr,EAX
  mov DL,[EBX+65536]

  test DL,DL
  jne Br4_Special

Br4_Continue:

  nop
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LDAb - Load accumulator (NZ), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             LDAb

proc(LDAb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AH,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne LDAb_Special

LDAb_Continue:

  mov [ESI]CPU65xx.bAccu,AH
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AH

  mov [ESI]CPU65xx.bZVal,AH
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LDA8 - Load accumulator (NZ), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   R  read data, perform action                       LDA8

proc(LDA8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov [ESI]CPU65xx.bAccu,AL

  cmp DL,CPU16x4k_iIOStart
  jae LDA8_Trap

LDA8_Continue:

  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL
  nop

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LDA16 - Load accumulator (NZ), a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       LDA16

proc(LDA16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  nop

  mov EBP,[ESI]Chip.pNextChip
  xor EDX,EDX

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae LDA16_Trap

LDA16_Continue:

  mov [ESI]CPU65xx.bAccu,AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// STA8 - Store accumulator (no flags), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   W  perform action, write data                      STA8

proc(STA8) index(0)

  mov EBP,ESI
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov ECX,[ESI]CPU65xx.iAddr8
  mov AL,[ESI]CPU65xx.bAccu

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]Chip.pfnOnClock,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// STA16 - Store accumulator (no flags), a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   W  perform action, write data                      STA16

proc(STA16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EBP,ESI

  shr EAX,12
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov ESI,[ESI]Chip.pNextChip
  mov [EBP]Chip.pfnOnClock,EDX

  mov EBX,[EBP]CPU16x4k.apbWriteMapping[EAX*4]
  mov ECX,[EBP]CPU65xx.iAddr16

  ;nop
  mov AL,[EBP]CPU65xx.bAccu

  mov DL,[EBX+ECX+65536]
  mov [EBX+ECX],AL

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LDXb - Load x register (NZ), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             LDXb

proc(LDXb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  xor EDX,EDX

  mov EBP,[ESI]Chip.pNextChip
  nop

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne LDXb_Special

LDXb_Continue:

  mov byte ptr ([ESI]CPU65xx.iXReg),AL
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL
  mov [ESI]Chip.pfnOnClock,EDX

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LDX8 - Load x register (NZ), z/zy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  3   00ll   R  read data, perform action                        LDX8

proc(LDX8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iXReg),AL

  cmp DL,CPU16x4k_iIOStart
  jae LDX8_Trap

LDX8_Continue:

  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL
  nop

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LDX16 - Load x register (NZ), a/ay/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       LDX16

proc(LDX16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  nop

  mov EBP,[ESI]Chip.pNextChip
  xor EDX,EDX

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae LDX16_Trap

LDX16_Continue:

  mov byte ptr ([ESI]CPU65xx.iXReg),AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// STX8 - Store x register (no flags), z/zy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   W  perform action, write data                      STX8

proc(STX8) index(0)

  mov EBP,ESI
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov ECX,[ESI]CPU65xx.iAddr8
  mov EAX,[ESI]CPU65xx.iXReg

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]Chip.pfnOnClock,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// STX16 - Store x register (no flags), a addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  4   hhll   W  perform action, write data                       STX16

proc(STX16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EBP,ESI

  shr EAX,12
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov ESI,[ESI]Chip.pNextChip
  mov [EBP]Chip.pfnOnClock,EDX

  mov EBX,[EBP]CPU16x4k.apbWriteMapping[EAX*4]
  mov ECX,[EBP]CPU65xx.iAddr16

  ;nop
  mov EAX,[EBP]CPU65xx.iXReg

  mov DL,[EBX+ECX+65536]
  mov [EBX+ECX],AL

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LDYb - Load y register (NZ), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             LDYb

proc(LDYb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  xor EDX,EDX

  mov EBP,[ESI]Chip.pNextChip
  nop

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne LDYb_Special

LDYb_Continue:

  mov byte ptr ([ESI]CPU65xx.iYReg),AL
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL
  mov [ESI]Chip.pfnOnClock,EDX

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LDY8 - Load y register (NZ), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   R  read data, perform action                       LDY8

proc(LDY8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iYReg),AL

  cmp DL,CPU16x4k_iIOStart
  jae LDY8_Trap

LDY8_Continue:

  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL
  nop

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LDY16 - Load y register (NZ), a/ax/ix addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       LDY16

proc(LDY16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  nop

  mov EBP,[ESI]Chip.pNextChip
  xor EDX,EDX

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae LDY16_Trap

LDY16_Continue:

  mov byte ptr ([ESI]CPU65xx.iYReg),AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// STY8 - Store y register (no flags), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   W  perform action, write data                      STY8

proc(STY8) index(0)

  mov EBP,ESI
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov ECX,[ESI]CPU65xx.iAddr8
  mov EAX,[ESI]CPU65xx.iYReg

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]Chip.pfnOnClock,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// STY16 - Store y register (no flags), a addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  4   hhll   W  perform action, write data                       STY16

proc(STY16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EBP,ESI

  shr EAX,12
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov ESI,[ESI]Chip.pNextChip
  mov [EBP]Chip.pfnOnClock,EDX

  mov EBX,[EBP]CPU16x4k.apbWriteMapping[EAX*4]
  mov ECX,[EBP]CPU65xx.iAddr16

  ;nop
  mov EAX,[EBP]CPU65xx.iYReg

  mov DL,[EBX+ECX+65536]
  mov [EBX+ECX],AL

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// TAXn - Transfer accu to x register (NZ), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     TAXn

proc(TAXn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov AL,[ESI]CPU65xx.bAccu

  mov EBP,[ESI]Chip.pNextChip
  mov byte ptr ([ESI]CPU65xx.iXReg),AL

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.bZVal,AL

  test DL,DL
  jne TAXn_Special

TAXn_Continue:

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// TAYn - Transfer accu to y register (NZ), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     TAYn

proc(TAYn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov AL,[ESI]CPU65xx.bAccu

  mov EBP,[ESI]Chip.pNextChip
  mov byte ptr ([ESI]CPU65xx.iYReg),AL

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.bZVal,AL

  test DL,DL
  jne TAYn_Special

TAYn_Continue:

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// TXAn - Transfer x register to accu (NZ), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     TXAn

proc(TXAn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]CPU65xx.bAccu,AL

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.bZVal,AL

  test DL,DL
  jne TXAn_Special

TXAn_Continue:

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// TYAn - Transfer y register to accu (NZ), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     TYAn

proc(TYAn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov AL,byte ptr ([ESI]CPU65xx.iYReg)

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]CPU65xx.bAccu,AL

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.bZVal,AL

  test DL,DL
  jne TYAn_Special

TYAn_Continue:

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// TSXn - Transfer SP to x register (NZ), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     TSXn

proc(TSXn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov AL,byte ptr ([ESI]CPU65xx.iSP)

  mov EBP,[ESI]Chip.pNextChip
  mov byte ptr ([ESI]CPU65xx.iXReg),AL

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.bZVal,AL

  test DL,DL
  jne TSXn_Special

TSXn_Continue:

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]Chip.pfnOnClock,EDX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// TXSn - Transfer x register to SP (no flags), no addressing
// Doesn't set N and Z flags!!!
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     TXSn

proc(TXSn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov [ESI]Chip.pfnOnClock,EDX
  mov EBP,[ESI]Chip.pNextChip

  mov DL,[EBX+65536]
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)

  test DL,DL
  jne TXSn_Special

TXSn_Continue:

  mov byte ptr ([ESI]CPU65xx.iSP),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// PHAn/PHPn - Push accu/push processor status (no flags), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch                                     PHx2
//  3   01SP   W  push register, SP -= 1                          pfnAction

proc(PHx2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnAction

  mov EBP,ESI
  mov [ESI]Chip.pfnOnClock,EDX

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+65536]

  test DL,DL
  jne PHx2_Special

PHx2_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(PHAn3) index(0)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov EBP,ESI

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX-1]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iSP),AL

  mov AL,[ESI]CPU65xx.bAccu
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(PHPn3) index(0)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov EBP,ESI

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX-1]

  mov byte ptr ([ESI]CPU65xx.iSP),AL
  mov EAX,[ESI]CPU65xx.iNCFlag

  mov DL,[ESI]CPU65xx.bVFlag
  and EAX,0000000110000000b

  shl DL,6
  or AL,AH

  mov DH,[ESI]CPU65xx.bZVal
  or AL,DL

  mov DL,[ESI]CPU65xx.bFlags
  cmp DH,0

  mov ESI,[ESI]Chip.pNextChip
  jne NoZeroPHPn3

  nop
  or AL,00000010b

NoZeroPHPn3:

  or AL,DL
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// PLAn/PLPn - Pull accu/pull processor status (NZ/all flags), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch                                     PLx2
//  2   01SP   R  dummy read, SP += 1                             PLx3
//  4   01SP   R  pull register                                   pfnAction

proc(PLx2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset PLx3+4

  mov EBP,ESI
  mov [ESI]Chip.pfnOnClock,EDX

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+65536]

  test DL,DL
  jne PLx2_Special

PLx2_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(PLx3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,[ESI]CPU65xx.pfnAction
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX+1]

  mov byte ptr ([ESI]CPU65xx.iSP),AL
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iDummyStart
  jae PLx3_Trap

PLx3_Continue:

  mov ESI,EBP
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(PLAn4) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov [ESI]CPU65xx.bAccu,AL

  cmp DL,CPU16x4k_iIOStart
  jae PLAn4_Trap

PLAn4_Continue:

  nop
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(PLPn4) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov AH,AL

  cmp DL,CPU16x4k_iIOStart
  jae PLPn4_Trap

PLPn4_Continue:

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov BL,AL

  shr AH,6
  not BL

  and EAX,0000000100111100b
  and BL,00000010b

  mov [ESI]CPU65xx.bVFlag,AH
  mov [ESI]CPU65xx.bZVal,BL

  or AL,00110000b
  mov AH,[ESI]CPU65xx.bIRQ

  mov [ESI]CPU65xx.bFlags,AL
  or AL,AH

  test AL,00000100b
  jne NoIRQPLPn4

  mov EAX,[ESI]CPU65xx.iInterrupts
  or EAX,giIRQ
  mov [ESI]CPU65xx.iInterrupts,EAX
  mov EAX,gapfnInterrupt[EAX*4]
  mov [ESI]CPU65xx.pfnInterrupt,EAX

NoIRQPLPn4:

  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// INXn - Increment x register (NZ), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     INXn

proc(INXn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EAX,[ESI]CPU65xx.iXReg

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  inc EAX

  mov [ESI]Chip.pfnOnClock,EDX
  mov byte ptr ([ESI]CPU65xx.iXReg),AL

  mov EBP,[ESI]Chip.pNextChip
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.bZVal,AL

  test DL,DL
  jne INXn_Special

INXn_Continue:

  nop
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// INYn - Increment y register (NZ), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     INYn

proc(INYn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EAX,[ESI]CPU65xx.iYReg

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  inc EAX

  mov [ESI]Chip.pfnOnClock,EDX
  mov byte ptr ([ESI]CPU65xx.iYReg),AL

  mov EBP,[ESI]Chip.pNextChip
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.bZVal,AL

  test DL,DL
  jne INYn_Special

INYn_Continue:

  nop
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// DEXn - Decrement x register (NZ), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     DEXn

proc(DEXn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EAX,[ESI]CPU65xx.iXReg

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  dec EAX

  mov [ESI]Chip.pfnOnClock,EDX
  mov byte ptr ([ESI]CPU65xx.iXReg),AL

  mov EBP,[ESI]Chip.pNextChip
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.bZVal,AL

  test DL,DL
  jne DEXn_Special

DEXn_Continue:

  nop
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// DEYn - Decrement y register (NZ), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     DEYn

proc(DEYn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EAX,[ESI]CPU65xx.iYReg

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  dec EAX

  mov [ESI]Chip.pfnOnClock,EDX
  mov byte ptr ([ESI]CPU65xx.iYReg),AL

  mov EBP,[ESI]Chip.pNextChip
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.bZVal,AL

  test DL,DL
  jne DEYn_Special

DEYn_Continue:

  nop
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// INC8 - Increment (NZ), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-5  00ll   W  write old data, perform action                  INC8

proc(INC8) index(0)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]
  mov DL,[ESI]CPU65xx.bData

  mov AL,DL
  inc DL
  
  mov ECX,[ESI]CPU65xx.iAddr8
  mov [ESI]CPU65xx.bData,DL

  mov EBP,offset Mz5+4
  mov [ESI]CPU65xx.bZVal,DL

  mov [ESI]Chip.pfnOnClock,EBP
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),DL

  mov DL,[EBX+ECX+65536]
  mov EBP,ESI

  mov [EBX+ECX],AL
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// INC16 - Increment (NZ), a/ax addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 5-6  hhll   W  write old data, perform action                  INC16

proc(INC16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov DL,[ESI]CPU65xx.bData

  shr EAX,12
  inc DL

  mov ECX,[ESI]CPU65xx.iAddr16
  mov EBP,offset Ma6+4

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EAX*4]
  mov AL,[ESI]CPU65xx.bData

  mov [ESI]CPU65xx.bData,DL
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.bZVal,DL
  mov EBP,ESI

  mov ESI,[ESI]Chip.pNextChip
  mov [EBX+ECX],AL

  mov byte ptr ([EBP]CPU65xx.iNCFlag[0]),DL
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// DEC8 - Decrement (NZ), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-5  00ll   W  write old data, perform action                  DEC8

proc(DEC8) index(0)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]
  mov DL,[ESI]CPU65xx.bData

  mov AL,DL
  dec DL
  
  mov ECX,[ESI]CPU65xx.iAddr8
  mov [ESI]CPU65xx.bData,DL

  mov EBP,offset Mz5+4
  mov [ESI]CPU65xx.bZVal,DL

  mov [ESI]Chip.pfnOnClock,EBP
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),DL

  mov DL,[EBX+ECX+65536]
  mov EBP,ESI

  mov [EBX+ECX],AL
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// DEC16 - Decrement (NZ), a/ax addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 5-6  hhll   W  write old data, perform action                  DEC16

proc(DEC16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov DL,[ESI]CPU65xx.bData

  shr EAX,12
  dec DL

  mov ECX,[ESI]CPU65xx.iAddr16
  mov EBP,offset Ma6+4

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EAX*4]
  mov AL,[ESI]CPU65xx.bData

  mov [ESI]CPU65xx.bData,DL
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.bZVal,DL
  mov EBP,ESI

  mov ESI,[ESI]Chip.pNextChip
  mov [EBX+ECX],AL

  mov byte ptr ([EBP]CPU65xx.iNCFlag[0]),DL
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ASLn - Arithmetic shift left (NZC), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     ASLn

proc(ASLn) index(0)

  mov AL,[ESI]CPU65xx.bAccu
  mov EBX,[ESI]CPU65xx.pbPCPtr

  shl EAX,1
  mov EBP,[ESI]Chip.pNextChip

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.bAccu,AL

  test DL,DL
  jne ASLn_Special

ASLn_Continue:

  mov [ESI]CPU65xx.bZVal,AL
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov [ESI]Chip.pfnOnClock,EDX

  mov ESI,EBP
  nop

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ASL8 - Arithmetic shift left (NZC), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-5  00ll   W  write old data, perform action                  ASL8

proc(ASL8) index(0)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]
  mov DL,[ESI]CPU65xx.bData

  shl EDX,1
  mov AL,[ESI]CPU65xx.bData

  mov ECX,[ESI]CPU65xx.iAddr8
  mov [ESI]CPU65xx.bData,DL

  mov EBP,offset Mz5+4
  mov [ESI]CPU65xx.bZVal,DL

  mov [ESI]Chip.pfnOnClock,EBP
  mov [ESI]CPU65xx.iNCFlag,EDX

  mov [EBX+ECX],AL
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ASL16 - Arithmetic shift left (NZC), a/ax addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 5-6  hhll   W  write old data, perform action                  ASL16

proc(ASL16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov DL,[ESI]CPU65xx.bData

  shr EAX,12
  mov ECX,[ESI]CPU65xx.iAddr16

  shl EDX,1
  mov EBP,offset Ma6+4

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EAX*4]
  mov AL,[ESI]CPU65xx.bData

  mov [ESI]CPU65xx.bData,DL
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.bZVal,DL
  mov EBP,ESI

  mov [ESI]CPU65xx.iNCFlag,EDX
  mov [EBX+ECX],AL

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LSRn - Logical shift right (NZC), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     LSRn

proc(LSRn) index(0)

  mov AL,[ESI]CPU65xx.bAccu
  mov EBX,[ESI]CPU65xx.pbPCPtr

  shr AL,1
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  sbb AH,AH
  mov [ESI]Chip.pfnOnClock,EDX

  mov [ESI]CPU65xx.bAccu,AL
  mov DL,[EBX+65536]

  test DL,DL
  jne LSRn_Special

LSRn_Continue:

  mov [ESI]CPU65xx.bZVal,AL
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LSR8 - Logical shift right (NZC), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-5  00ll   W  write old data, perform action                  LSR8

proc(LSR8) index(0)

  mov DL,[ESI]CPU65xx.bData
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  shr DL,1
  mov AL,[ESI]CPU65xx.bData

  sbb DH,DH
  mov ECX,[ESI]CPU65xx.iAddr8

  mov [ESI]CPU65xx.bData,DL
  mov EDI,offset Mz5+4

  mov EBP,ESI
  mov [ESI]Chip.pfnOnClock,EDI

  mov [ESI]CPU65xx.bZVal,DL
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.iNCFlag,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LSR16 - Logical shift right (NZC), a/ax addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 5-6  hhll   W  write old data, perform action                  LSR16

proc(LSR16) index(0)

  mov DL,[ESI]CPU65xx.bData
  mov EAX,[ESI]CPU65xx.iAddr16

  shr EAX,12
  mov ECX,[ESI]CPU65xx.iAddr16

  shr DL,1
  mov EBP,offset Ma6+4

  sbb DH,DH
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EAX*4]

  mov AL,[ESI]CPU65xx.bData
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.bData,DL
  mov EBP,ESI

  mov [ESI]CPU65xx.bZVal,DL
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.iNCFlag,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ROLn - Rotate left (NZC), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     ROLn

proc(ROLn) index(0)

  mov AL,[ESI]CPU65xx.bAccu
  mov EBX,[ESI]CPU65xx.pbPCPtr

  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr AH,1
  mov [ESI]Chip.pfnOnClock,EDX

  adc EAX,EAX
  mov EBP,[ESI]Chip.pNextChip

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.bAccu,AL

  test DL,DL
  jne ROLn_Special

ROLn_Continue:

  mov [ESI]CPU65xx.bZVal,AL
  nop

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ROL8 - Rotate left (NZC), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-5  00ll   W  write old data, perform action                  ROL8

proc(ROL8) index(0)

  mov DL,[ESI]CPU65xx.bData
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov DH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov ECX,[ESI]CPU65xx.iAddr8

  shr DH,1
  mov EBP,offset Mz5+4

  adc EDX,EDX
  mov AL,[ESI]CPU65xx.bData

  mov [ESI]CPU65xx.bData,DL
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.bZVal,DL
  mov EBP,ESI

  mov [EBX+ECX],AL
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.iNCFlag,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ROL16 - Rotate left (NZC), a/ax addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 5-6  hhll   W  write old data, perform action                  ROL16

proc(ROL16) index(0)

  mov DL,[ESI]CPU65xx.bData
  mov EAX,[ESI]CPU65xx.iAddr16

  shr EAX,12
  mov DH,byte ptr ([ESI]CPU65xx.iNCFlag[1])

  shr DH,1
  mov ECX,[ESI]CPU65xx.iAddr16

  adc EDX,EDX
  mov EBP,offset Ma6+4

  mov [ESI]CPU65xx.bZVal,DL
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EAX*4]

  mov AL,[ESI]CPU65xx.bData
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.bData,DL
  mov EBP,ESI

  mov [EBX+ECX],AL
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.iNCFlag,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp



////////////////////////////////////////////////////////////////////////////////
// RORn - Rotate right (NZC), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     RORn

proc(RORn) index(0)

  mov AL,[ESI]CPU65xx.bAccu
  mov EBX,[ESI]CPU65xx.pbPCPtr

  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,1
  mov [ESI]Chip.pfnOnClock,EDX

  sbb AH,AH
  mov EBP,[ESI]Chip.pNextChip

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.bAccu,AL

  test DL,DL
  jne RORn_Special

RORn_Continue:

  mov [ESI]CPU65xx.bZVal,AL
  nop

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ROR8 - Rotate right (NZC), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-5  00ll   W  write old data, perform action                  ROR8

proc(ROR8) index(0)

  mov DL,[ESI]CPU65xx.bData
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov DH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov ECX,[ESI]CPU65xx.iAddr8

  shr EDX,1
  mov EBP,offset Mz5+4

  sbb DH,DH
  mov AL,[ESI]CPU65xx.bData

  mov [ESI]CPU65xx.bData,DL
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.bZVal,DL
  mov EBP,ESI

  mov [EBX+ECX],AL
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.iNCFlag,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ROR16 - Rotate right (NZC), a/ax addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 5-6  hhll   W  write old data, perform action                  ROR16

proc(ROR16) index(0)

  mov DL,[ESI]CPU65xx.bData
  mov EAX,[ESI]CPU65xx.iAddr16

  shr EAX,12
  mov DH,byte ptr ([ESI]CPU65xx.iNCFlag[1])

  shr EDX,1
  mov ECX,[ESI]CPU65xx.iAddr16

  sbb DH,DH
  mov EBP,offset Ma6+4

  mov [ESI]CPU65xx.bZVal,DL
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EAX*4]

  mov AL,[ESI]CPU65xx.bData
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.bData,DL
  mov EBP,ESI

  mov [EBX+ECX],AL
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.iNCFlag,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ANDb - AND (NZ), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             ANDb

proc(ANDb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AH,[ESI]CPU65xx.bAccu
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne ANDb_Special

ANDb_Continue:

  and AL,AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bAccu,AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// AND8 - AND (NZ), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   R  read data, perform action                       AND8

proc(AND8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov AH,[ESI]CPU65xx.bAccu

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  nop

  cmp DL,CPU16x4k_iIOStart
  jae AND8_Trap

AND8_Continue:

  and AL,AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bAccu,AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// AND16 - AND (NZ), a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       AND16

proc(AND16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  mov AH,[ESI]CPU65xx.bAccu

  ;nop
  ;nop

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae AND16_Trap

AND16_Continue:

  and AL,AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bAccu,AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ORAb - OR accumulator (NZ), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             ORAb

proc(ORAb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AH,[ESI]CPU65xx.bAccu
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne ORAb_Special

ORAb_Continue:

  or AL,AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bAccu,AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ORA8 - OR accumulator (NZ), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   R  read data, perform action                       ORA8

proc(ORA8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov AH,[ESI]CPU65xx.bAccu

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  nop

  cmp DL,CPU16x4k_iIOStart
  jae ORA8_Trap

ORA8_Continue:

  or AL,AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bAccu,AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ORA16 - OR accumulator (NZ), a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       ORA16

proc(ORA16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  mov AH,[ESI]CPU65xx.bAccu

  ;nop
  ;nop

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae ORA16_Trap

ORA16_Continue:

  or AL,AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bAccu,AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// EORb - Exclusive OR (NZ), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             EORb

proc(EORb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AH,[ESI]CPU65xx.bAccu
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne EORb_Special

EORb_Continue:

  xor AL,AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bAccu,AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// EOR8 - Exclusive OR (NZ), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   R  read data, perform action                       EOR8

proc(EOR8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov AH,[ESI]CPU65xx.bAccu

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  nop

  cmp DL,CPU16x4k_iIOStart
  jae EOR8_Trap

EOR8_Continue:

  xor AL,AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bAccu,AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// EOR16 - Exclusive OR (NZ), a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       EOR16

proc(EOR16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  mov AH,[ESI]CPU65xx.bAccu

  ;nop
  ;nop

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae EOR16_Trap

EOR16_Continue:

  xor AL,AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bAccu,AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CLCn - Clear carry (C), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     CLCn

proc(CLCn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+65536]
  xor AL,AL

  test DL,DL
  jne CLCn_Special

CLCn_Continue:

  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// SECn - Set carry (C), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     SECn

proc(SECn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+65536]
  mov AL,1

  test DL,DL
  jne SECn_Special

SECn_Continue:

  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CLDn - Clear decimal flag (D), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     CLDn

proc(CLDn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+65536]
  mov AL,[ESI]CPU65xx.bFlags

  test DL,DL
  jne CLDn_Special

CLDn_Continue:

  and AL,~0x08
  nop

  mov [ESI]CPU65xx.bFlags,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// SEDn - Set decimal flag (D), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     SEDn

proc(SEDn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+65536]
  mov AL,[ESI]CPU65xx.bFlags

  test DL,DL
  jne SEDn_Special

SEDn_Continue:

  or AL,0x08
  nop

  mov [ESI]CPU65xx.bFlags,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CLIn - Clear interrupt flag (IRQ interrupts on) (I), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     CLIn

proc(CLIn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+65536]
  mov AL,[ESI]CPU65xx.bFlags

  test DL,DL
  jne CLIn_Special

CLIn_Continue:

  and AL,~0x04
  mov EBX,[ESI]CPU65xx.iInterrupts

  or EBX,giIRQ
  mov CL,[ESI]CPU65xx.bIRQ

  test CL,CL
  jne CLIn_NoIRQ

  mov [ESI]CPU65xx.iInterrupts,EBX
  mov ECX,gapfnInterrupt[EBX*4]

  mov [ESI]CPU65xx.pfnInterrupt,ECX
  ;nop

CLIn_NoIRQ:

  mov [ESI]CPU65xx.bFlags,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// SEIn - Set interrupt flag (IRQ interrupts off) (I), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     SEIn

proc(SEIn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+65536]
  mov AL,[ESI]CPU65xx.bFlags

  test DL,DL
  jne SEIn_Special

SEIn_Continue:

  or AL,0x04
  mov AH,[ESI]CPU65xx.bIRQ

  mov [ESI]CPU65xx.bFlags,AL
  test EAX,0x0400

  mov EAX,[ESI]CPU65xx.iInterrupts
  jne SEIn_NoIRQPending

  and EAX,~giIRQ
  mov [ESI]CPU65xx.iInterrupts,EAX
  mov EAX,gapfnInterrupt[EAX*4]
  mov [ESI]CPU65xx.pfnInterrupt,EAX

SEIn_NoIRQPending:

  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CLVn - Clear overflow (V), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     CLVn

proc(CLVn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+65536]
  xor AL,AL

  test DL,DL
  jne CLVn_Special

CLVn_Continue:

  mov [ESI]CPU65xx.bVFlag,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ADCb - Add with carry (NZCV), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             ADCb

proc(ADCb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AH,[ESI]CPU65xx.bFlags
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne ADCb_Special

ADCb_Continue:

  test EAX,0x0800
  jne AddDec

  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov BL,[ESI]CPU65xx.bAccu

  shr AH,1
  mov EBP,ESI

  adc BL,AL
  mov ESI,[ESI]Chip.pNextChip

  ;two
  ;clocks

  ;unpaired
  seto AL

  sbb BH,BH
  mov [EBP]CPU65xx.bVFlag,AL

  mov [EBP]CPU65xx.bAccu,BL
  nop

  mov [EBP]CPU65xx.iNCFlag,EBX
  mov [EBP]CPU65xx.bZVal,BL

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ADC8 - Add with carry (NZCV), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   R  read data, perform action                       ADC8

proc(ADC8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov AH,[ESI]CPU65xx.bFlags

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  nop

  cmp DL,CPU16x4k_iIOStart
  jae ADC8_Trap

ADC8_Continue:

  test EAX,0x0800
  jne AddDec

  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov BL,[ESI]CPU65xx.bAccu

  shr AH,1
  mov EBP,ESI

  adc BL,AL
  mov ESI,[ESI]Chip.pNextChip

  ;two
  ;clocks

  ;unpaired
  seto AL

  sbb BH,BH
  mov [EBP]CPU65xx.bVFlag,AL

  mov [EBP]CPU65xx.bAccu,BL
  nop

  mov [EBP]CPU65xx.iNCFlag,EBX
  mov [EBP]CPU65xx.bZVal,BL

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ADC16 - Add with carry (NZCV), a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       ADC16

proc(ADC16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov ECX,[ESI]CPU65xx.iAddr16
  mov [ESI]Chip.pfnOnClock,EDX

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  mov AH,[ESI]CPU65xx.bFlags

  ;nop
  ;nop

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae ADC16_Trap

ADC16_Continue:

  test EAX,0x0800
  jne AddDec

  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov BL,[ESI]CPU65xx.bAccu

  shr AH,1
  mov EBP,ESI

  adc BL,AL
  mov ESI,[ESI]Chip.pNextChip

  ;two
  ;clocks

  ;unpaired
  seto AL

  sbb BH,BH
  mov [EBP]CPU65xx.bVFlag,AL

  mov [EBP]CPU65xx.bAccu,BL
  nop

  mov [EBP]CPU65xx.iNCFlag,EBX
  mov [EBP]CPU65xx.bZVal,BL

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// SBCb - Subtract with carry (NZCV), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             SBCb

proc(SBCb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AH,[ESI]CPU65xx.bFlags
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne SBCb_Special

SBCb_Continue:

  test EAX,0x0800
  jne SubDec

  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov BL,[ESI]CPU65xx.bAccu

  not AH
  nop

  shr AH,1
  mov EBP,ESI

  sbb BL,AL
  mov ESI,[ESI]Chip.pNextChip

  ;two
  ;clocks

  ;unpaired
  seto AL

  sbb BH,BH
  mov [EBP]CPU65xx.bVFlag,AL

  mov [EBP]CPU65xx.bAccu,BL
  not BH

  mov [EBP]CPU65xx.iNCFlag,EBX
  mov [EBP]CPU65xx.bZVal,BL

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// SBC8 - Subtract with carry (NZCV), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   R  read data, perform action                       SBC8

proc(SBC8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov AH,[ESI]CPU65xx.bFlags

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  nop

  cmp DL,CPU16x4k_iIOStart
  jae SBC8_Trap

SBC8_Continue:

  test EAX,0x0800
  jne SubDec

  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov BL,[ESI]CPU65xx.bAccu

  not AH
  nop

  shr AH,1
  mov EBP,ESI

  sbb BL,AL
  mov ESI,[ESI]Chip.pNextChip

  ;two
  ;clocks

  ;unpaired
  seto AL

  sbb BH,BH
  mov [EBP]CPU65xx.bVFlag,AL

  mov [EBP]CPU65xx.bAccu,BL
  not BH

  mov [EBP]CPU65xx.iNCFlag,EBX
  mov [EBP]CPU65xx.bZVal,BL

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// SBC16 - Subtract with carry (NZCV), a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       SBC16

proc(SBC16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  mov AH,[ESI]CPU65xx.bFlags

  ;nop
  ;nop

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae SBC16_Trap

SBC16_Continue:

  test EAX,0x0800
  jne SubDec

  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov BL,[ESI]CPU65xx.bAccu

  not AH
  nop

  shr AH,1
  mov EBP,ESI

  sbb BL,AL
  mov ESI,[ESI]Chip.pNextChip

  ;two
  ;clocks

  ;unpaired
  seto AL

  sbb BH,BH
  mov [EBP]CPU65xx.bVFlag,AL

  mov [EBP]CPU65xx.bAccu,BL
  not BH

  mov [EBP]CPU65xx.iNCFlag,EBX
  mov [EBP]CPU65xx.bZVal,BL

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Add/subtract in decimal mode
// AL  = data
// ESI = CPU65xx
// EBP = pNextObject

proc(AddDec)

  mov DL,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov AH,[ESI]CPU65xx.bAccu

  shr DL,1
  mov ECX,EAX

  adc CH,CL
  mov EBX,EAX

  mov DL,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  and EAX,0x0F0F

  shr DL,1
  mov [ESI]CPU65xx.bZVal,CH

  adc AH,AL
  and EBX,0xF0F0

  cmp AH,0x0A
  jb AddDec_LowOK

  sub AH,0x0A
  or EBX,0x0808

AddDec_LowOK:

  add BH,BL
  mov CL,1

  ;two
  ;clocks

  ;unpaired
  seto DH

  sbb DL,DL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),BH

  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),DL
  jne AddDec_AdjustHigh

  cmp BH,0xA0
  jb AddDec_HighOK

AddDec_AdjustHigh:

  sub BH,0xA0
  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),CL

AddDec_HighOK:

  and AH,0x0F
  mov [ESI]CPU65xx.bVFlag,DH

  or AH,BH
  nop

  mov [ESI]CPU65xx.bAccu,AH
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(SubDec)

  mov DL,byte ptr ([ESI]CPU65xx.iNCFlag)[1]
  mov AH,[ESI]CPU65xx.bAccu

  not DL
  mov ECX,EAX

  shr DL,1
  mov EBX,EAX

  sbb CH,CL
  mov DL,byte ptr ([ESI]CPU65xx.iNCFlag[1])

  not DL
  and EAX,0x0F0F

  and EBX,0xF0F0
  nop

  shr DL,1
  mov [ESI]CPU65xx.bZVal,CH

  sbb AH,AL
  jnc SubDec_LowOK

  sub AH,0x06
  or BL,0x08

SubDec_LowOK:

  sub BH,BL
  mov CL,0

  ;two
  ;clocks

  ;unpaired
  seto CH

  sbb DL,DL
  mov [ESI]CPU65xx.bVFlag,CH

  not DL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),BH

  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),DL
  je SubDec_HighOK

  sub BH,0x60
  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),CL

SubDec_HighOK:

  and BH,0xF0
  and AH,0x0F

  or AH,BH
  nop

  mov [ESI]CPU65xx.bAccu,AH
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CMPb - Compare (NZC), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             CMPb

proc(CMPb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AL,[ESI]CPU65xx.bAccu
  mov [ESI]Chip.pfnOnClock,EDX

  mov AH,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne CMPb_Special

CMPb_Continue:

  sub AL,AH
  nop

  sbb AH,AH
  mov [ESI]CPU65xx.bZVal,AL

  not AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CMP8 - Compare (NZC), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   R  read data, perform action                       CMP8

proc(CMP8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov AL,[ESI]CPU65xx.bAccu

  mov [ESI]Chip.pfnOnClock,EDX
  mov AH,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  nop

  cmp DL,CPU16x4k_iIOStart
  jae CMP8_Trap

CMP8_Continue:

  sub AL,AH
  nop

  sbb AH,AH
  mov [ESI]CPU65xx.bZVal,AL

  not AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CMP16 - Compare (NZC), a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       CMP16

proc(CMP16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  nop

  mov EBP,[ESI]Chip.pNextChip
  mov AL,[ESI]CPU65xx.bAccu

  mov AH,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae CMP16_Trap

CMP16_Continue:

  sub AL,AH
  nop

  sbb AH,AH
  mov [ESI]CPU65xx.bZVal,AL

  not AH
  nop

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CPXb - Compare x register (NZC), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             CPXb

proc(CPXb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AL,byte ptr ([ESI]CPU65xx.iXReg)
  mov [ESI]Chip.pfnOnClock,EDX

  mov AH,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne CPXb_Special

CPXb_Continue:

  sub AL,AH
  nop

  sbb AH,AH
  mov [ESI]CPU65xx.bZVal,AL

  not AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CPX8 - Compare x register (NZC), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   R  read data, perform action                       CPX8

proc(CPX8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)

  mov [ESI]Chip.pfnOnClock,EDX
  mov AH,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  nop

  cmp DL,CPU16x4k_iIOStart
  jae CPX8_Trap

CPX8_Continue:

  sub AL,AH
  nop

  sbb AH,AH
  mov [ESI]CPU65xx.bZVal,AL

  not AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CPX16 - Compare x register (NZC), a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       CPX16

proc(CPX16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  nop

  mov EBP,[ESI]Chip.pNextChip
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)

  mov AH,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae CPX16_Trap

CPX16_Continue:

  sub AL,AH
  nop

  sbb AH,AH
  mov [ESI]CPU65xx.bZVal,AL

  not AH
  nop

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CPYb - Compare y register (NZC), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             CPYb

proc(CPYb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AL,byte ptr ([ESI]CPU65xx.iYReg)
  mov [ESI]Chip.pfnOnClock,EDX

  mov AH,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne CPYb_Special

CPYb_Continue:

  sub AL,AH
  nop

  sbb AH,AH
  mov [ESI]CPU65xx.bZVal,AL

  not AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CPY8 - Compare y register (NZC), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   R  read data, perform action                       CPY8

proc(CPY8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov AL,byte ptr ([ESI]CPU65xx.iYReg)

  mov [ESI]Chip.pfnOnClock,EDX
  mov AH,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  nop

  cmp DL,CPU16x4k_iIOStart
  jae CPY8_Trap

CPY8_Continue:

  sub AL,AH
  nop

  sbb AH,AH
  mov [ESI]CPU65xx.bZVal,AL

  not AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// CPY16 - Compare y register (NZC), a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       CPY16

proc(CPY16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  nop

  mov EBP,[ESI]Chip.pNextChip
  mov AL,byte ptr ([ESI]CPU65xx.iYReg)

  mov AH,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae CPY16_Trap

CPY16_Continue:

  sub AL,AH
  nop

  sbb AH,AH
  mov [ESI]CPU65xx.bZVal,AL

  not AH
  nop

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// BIT8 - Bit test (Z, N=7, V=6), z addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  3   00ll   R  read data, perform action                       BIT8

proc(BIT8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov AH,[ESI]CPU65xx.bAccu

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  nop

  cmp DL,CPU16x4k_iIOStart
  jae BIT8_Trap

BIT8_Continue:

  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL
  test AL,0x40

  ;unpaired
  setne BL

  and AL,AH
  mov [ESI]CPU65xx.bVFlag,BL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// BIT16 - Bit test (Z, N=7, V=6), a addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  4   hhll   R  read data, perform action                       BIT16

proc(BIT16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  nop

  mov EBP,[ESI]Chip.pNextChip
  mov AH,[ESI]CPU65xx.bAccu

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae BIT16_Trap

BIT16_Continue:

  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL
  test AL,0x40

  ;unpaired
  setne BL

  and AL,AH
  mov [ESI]CPU65xx.bVFlag,BL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// BRKn - Break (no flags), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, PC += 1                            BRKn2
//  3   01SP   W  push PCH, SP -= 1                               BRKn3
//  4   01SP   W  push PCL, SP -= 1                               BRKn4
//  5   01SP   W  push flags, SP -= 1, set IF                     BRKn5
//  6   FFFE   R  read ll                                         BRKn6
//  7   FFFF   R  read hh, PC = hhll                              BRKn7
//
// If NMI is active before clock 4, jump to NMI sequence instead
// but do not clear BFlag on stack.

proc(BRKn2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset BRKn3+4

  mov EBP,ESI
  mov [ESI]Chip.pfnOnClock,EDX

  lea ECX,[EBX+1]
  mov ESI,[ESI]Chip.pNextChip
 
  mov [EBP]CPU65xx.pbPCPtr,ECX
  mov DL,[EBX+65536]

  test DL,DL
  jne BRKn2_Special

BRKn2_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(BRKn3) index(0)

  mov EDX,offset BRKn4+4
  mov ECX,[ESI]CPU65xx.iSP

  mov [ESI]Chip.pfnOnClock,EDX
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov EBP,ESI
  lea EAX,[ECX-1]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iSP),AL

  mov EAX,[ESI]CPU65xx.pbPCPtr
  mov EDI,[ESI]CPU65xx.pbPCBase
  
  sub EAX,EDI
  mov ESI,[ESI]Chip.pNextChip

  mov [EBX+ECX],AH
  mov AL,AH

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(BRKn4) index(0)

  mov EAX,[ESI]CPU65xx.iInterrupts
  ;nop

  test EAX,giNMI
  jne jmpBRKNMI4_4

  mov EDX,offset BRKn5+4
  mov ECX,[ESI]CPU65xx.iSP

  mov [ESI]Chip.pfnOnClock,EDX
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov EBP,ESI
  lea EAX,[ECX-1]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iSP),AL

  mov EAX,[ESI]CPU65xx.pbPCPtr
  mov EDI,[ESI]CPU65xx.pbPCBase

  sub EAX,EDI
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(BRKn5) index(0)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,offset BRKn6+4
  mov EBP,ESI

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX-1]

  mov byte ptr ([ESI]CPU65xx.iSP),AL
  mov EAX,[ESI]CPU65xx.iNCFlag

  mov DL,[ESI]CPU65xx.bVFlag
  and EAX,0000000110000000b

  shl DL,6
  or AL,AH

  mov DH,[ESI]CPU65xx.bZVal
  or AL,DL

  mov DL,[ESI]CPU65xx.bFlags
  mov ESI,[ESI]Chip.pNextChip

  cmp DH,0
  jne BRKn5_NoZero

  or AL,00000010b
  ;nop

BRKn5_NoZero:

  or AL,DL
  or DL,0x04

  mov [EBP]CPU65xx.bFlags,DL
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(BRKn6) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[(0xFFFE shr 12)*4]
  mov EDX,offset BRKn7+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+0xFFFE+65536]
  mov AL,[EBX+0xFFFE]

  cmp DL,CPU16x4k_iIOStart
  jae BRKn6_Trap

BRKn6_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(BRKn7) index(0)

  mov EDX,offset Command+4
  mov EBX,[ESI]CPU16x4k.apbReadMapping[(0xFFFF shr 12)*4]

  mov [ESI]Chip.pfnOnClock,EDX
  nop

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EBP,[ESI]Chip.pNextChip

  mov DL,[EBX+0xFFFF+65536]
  mov AH,[EBX+0xFFFF]

  cmp DL,CPU16x4k_iIOStart
  jae BRKn7_Trap

BRKn7_Continue:

  mov EBX,EAX
  shr EBX,12
  mov EBX,[ESI]CPU16x4k.apbReadMapping[EBX*4]
  mov [ESI]CPU65xx.pbPCBase,EBX
  add EAX,EBX
  mov [ESI]CPU65xx.pbPCPtr,EAX
  mov ESI,EBP
  jmp [EBP]Chip.pfnOnClock

endp

jmpBRKNMI4_4: // TODO: remove
  jmp BRKNMI4+4

proc(BRKNMI4) index(0)

  and EAX,~giNMI
  mov EDX,offset BRKNMI5+4

  mov [ESI]CPU65xx.iInterrupts,EAX
  mov ECX,[ESI]CPU65xx.iSP

  mov EAX,gapfnInterrupt[EAX*4]
  mov [ESI]Chip.pfnOnClock,EDX

  mov [ESI]CPU65xx.pfnInterrupt,EAX
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov EBP,ESI
  lea EAX,[ECX-1]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iSP),AL

  mov EAX,[ESI]CPU65xx.pbPCPtr
  mov EDI,[ESI]CPU65xx.pbPCBase

  sub EAX,EDI
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(BRKNMI5) index(0)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,offset NMI6+4
  mov EBP,ESI

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX-1]

  mov byte ptr ([ESI]CPU65xx.iSP),AL
  mov EAX,[ESI]CPU65xx.iNCFlag

  mov DL,[ESI]CPU65xx.bVFlag
  and EAX,0000000110000000b

  shl DL,6
  or AL,AH

  mov DH,[ESI]CPU65xx.bZVal
  or AL,DL

  mov DL,[ESI]CPU65xx.bFlags
  mov ESI,[ESI]Chip.pNextChip

  cmp DH,0
  jne BRKNMI5_NoZero

  or AL,00000010b
  ;nop

BRKNMI5_NoZero:

  or AL,DL
  or DL,0x04

  mov [EBP]CPU65xx.bFlags,DL
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp







////////////////////////////////////////////////////////////////////////////////
// RTIn - Return from interrupt (all flags), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch                                     RTIn2
//  3   01SP   R  dummy read, SP += 1                             RTIn3
//  4   01SP   R  pull flags, SP += 1                             RTIn4
//  5   01SP   R  pull PCL, SP += 1                               RTIn5
//  6   01SP   R  pull PCH                                        RTIn6

proc(RTIn2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset RTIn3+4

  mov EBP,ESI
  mov [ESI]Chip.pfnOnClock,EDX

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+65536]

  test DL,DL
  jne RTIn2_Special

RTIn2_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(RTIn3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,offset RTIn4+4
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX+1]

  mov byte ptr ([ESI]CPU65xx.iSP),AL
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iDummyStart
  jae RTIn3_Trap

RTIn3_Continue:

  nop
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(RTIn4) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EBP,[ESI]Chip.pNextChip
  mov EDX,offset RTIn5+4

  lea EAX,[ECX+1]
  mov [ESI]Chip.pfnOnClock,EDX

  mov byte ptr ([ESI]CPU65xx.iSP),AL
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov AH,AL

  cmp DL,CPU16x4k_iIOStart
  jae RTIn4_Trap

RTIn4_Continue:

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov BL,AL

  shr AH,6
  not BL

  and EAX,0000000100111100b
  and BL,00000010b

  mov [ESI]CPU65xx.bVFlag,AH
  mov [ESI]CPU65xx.bZVal,BL

  or AL,00110000b
  mov BL,[ESI]CPU65xx.bIRQ

  mov [ESI]CPU65xx.bFlags,AL
  or AL,BL

  test AL,00000100b
  jne RTIn4_NoIRQ

  mov EAX,[ESI]CPU65xx.iInterrupts
  or EAX,giIRQ
  mov [ESI]CPU65xx.iInterrupts,EAX
  mov EAX,gapfnInterrupt[EAX*4]
  mov [ESI]CPU65xx.pfnInterrupt,EAX

RTIn4_NoIRQ:

  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(RTIn5) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,offset RTIn6+4
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX+1]

  nop
  mov byte ptr ([ESI]CPU65xx.iSP),AL

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae RTIn5_Trap

RTIn5_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(RTIn6) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),AL

  cmp DL,CPU16x4k_iIOStart
  jae RTIn6_Trap

RTIn6_Continue:

  mov EBX,[ESI]CPU65xx.iAddr16
  shr EBX,12
  mov EAX,[ESI]CPU65xx.iAddr16
  mov EBX,[ESI]CPU16x4k.apbReadMapping[EBX*4]
  mov [ESI]CPU65xx.pbPCBase,EBX
  add EAX,EBX
  mov [ESI]CPU65xx.pbPCPtr,EAX

  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// JSRw - Jump subroutine (no flags), word addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       JSRw2
//  3   01SP   R  dummy read                                      JSRw3
//  4   01SP   W  push PCH, SP -= 1                               JSRw4
//  5   01SP   W  push PCL, SP -= 1                               JSRw5
//  6    PC    R  fetch address hh, PC = hhll                     JSRw6

proc(JSRw2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset JSRw3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne JSRw2_Special

JSRw2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(JSRw3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,offset JSRw4+4
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iDummyStart
  jae JSRw3_Trap

JSRw3_Continue:

  nop
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(JSRw4) index(0)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,offset JSRw5+4
  mov EBP,ESI

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX-1]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iSP),AL

  mov EAX,[ESI]CPU65xx.pbPCPtr
  mov EDI,[ESI]CPU65xx.pbPCBase

  sub EAX,EDI
  mov ESI,[ESI]Chip.pNextChip

  mov [EBX+ECX],AH
  mov AL,AH

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(JSRw5) index(0)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,offset JSRw6+4
  mov EBP,ESI

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX-1]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iSP),AL

  mov EAX,[ESI]CPU65xx.pbPCPtr
  mov EDI,[ESI]CPU65xx.pbPCBase

  sub EAX,EDI
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(JSRw6) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EAX,[ESI]CPU65xx.iAddr16
  mov [ESI]Chip.pfnOnClock,EDX

  mov AH,[EBX]
  mov DL,[EBX+65536]

  test DL,DL
  jne JSRw6_Special

JSRw6_Continue:

  mov EBP,[ESI]Chip.pNextChip
  mov EBX,EAX
  shr EBX,12
  mov EBX,[ESI]CPU16x4k.apbReadMapping[EBX*4]
  mov [ESI]CPU65xx.pbPCBase,EBX
  add EAX,EBX
  mov [ESI]CPU65xx.pbPCPtr,EAX

  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// RTSn - Return from subroutine (no flags), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch                                     RTSn2
//  3   01SP   R  dummy read, SP += 1                             RTSn3
//  4   01SP   R  pull PCL, SP += 1                               RTSn4
//  5   01SP   R  pull PCH                                        RTSn5
//  6    PC    R  dummy fetch, PC += 1                            RTSn6

proc(RTSn2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset RTSn3+4

  mov EBP,ESI
  mov [ESI]Chip.pfnOnClock,EDX

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+65536]

  test DL,DL
  jne RTSn2_Special

RTSn2_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp

proc(RTSn3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,offset RTSn4+4
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX+1]

  mov byte ptr ([ESI]CPU65xx.iSP),AL
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iDummyStart
  jae RTSn3_Trap

RTSn3_Continue:

  nop
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(RTSn4) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,offset RTSn5+4
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  lea EAX,[ECX+1]

  nop
  mov byte ptr ([ESI]CPU65xx.iSP),AL

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae RTSn4_Trap

RTSn4_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(RTSn5) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iSP

  mov EDX,offset RTSn6+4
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),AL

  cmp DL,CPU16x4k_iIOStart
  jae RTSn5_Trap

RTSn5_Continue:

  mov EBX,[ESI]CPU65xx.iAddr16
  shr EBX,12
  mov EAX,[ESI]CPU65xx.iAddr16
  mov EBX,[ESI]CPU16x4k.apbReadMapping[EBX*4]
  mov [ESI]CPU65xx.pbPCBase,EBX
  add EAX,EBX
  mov [ESI]CPU65xx.pbPCPtr,EAX

  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(RTSn6) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EBP,ESI
  mov [ESI]Chip.pfnOnClock,EDX

  lea ECX,[EBX+1]
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.pbPCPtr,ECX
  mov DL,[EBX+65536]

  test DL,DL
  jne RTSn6_Special

RTSn6_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// JMPw - Jump (no flags), word addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       JMPw2
//  3    PC    R  fetch address hh, PC = hhll                     JMPw3

proc(JMPw2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset JMPw3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne JMPw2_Special

JMPw2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(JMPw3) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EAX,[ESI]CPU65xx.iAddr16
  mov [ESI]Chip.pfnOnClock,EDX

  mov AH,[EBX]
  mov DL,[EBX+65536]

  test DL,DL
  jne JMPw3_Special

JMPw3_Continue:

  mov EBP,[ESI]Chip.pNextChip
  mov EBX,EAX
  shr EBX,12
  mov EBX,[ESI]CPU16x4k.apbReadMapping[EBX*4]
  mov [ESI]CPU65xx.pbPCBase,EBX
  add EAX,EBX
  mov [ESI]CPU65xx.pbPCPtr,EAX

  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// JMPi - Jump (no flags), indirect addressing
// because hh is not incremented, JMP (xxFF) reads the high byte from xx00!
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       JMPi2
//  3    PC    R  fetch address hh, PC += 1                       JMPi3
//  4   hhll   R  read address xx, ll += 1                        JMPi4
//  5   hhll   R  read address yy, PC = yyxx                      JMPi5

proc(JMPi2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset JMPi3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne JMPi2_Special

JMPi2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(JMPi3) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset JMPi4+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne JMPi3_Special

JMPi3_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(JMPi4) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,offset JMPi5+4

  shr EAX,12
  mov ECX,[ESI]CPU65xx.iAddr16

  nop
  mov [ESI]Chip.pfnOnClock,EDX

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  lea EDX,[ECX+1]

  mov EBP,[ESI]Chip.pNextChip
  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),DL

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae JMPi4_Trap

JMPi4_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(JMPi5) index(0)

  mov EBX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EBX,12
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EAX,[ESI]CPU65xx.iAddr8
  mov [ESI]Chip.pfnOnClock,EDX

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EBX*4]
  mov EBP,[ESI]Chip.pNextChip

  mov AH,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae JMPi5_Trap

JMPi5_Continue:

  mov EBX,EAX
  shr EBX,12
  mov EBX,[ESI]CPU16x4k.apbReadMapping[EBX*4]
  mov [ESI]CPU65xx.pbPCBase,EBX
  add EAX,EBX
  mov [ESI]CPU65xx.pbPCPtr,EAX

  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// NOPn - No operation (no flags), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  dummy fetch, perform action                     NOPn

proc(NOPn) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt
  
  mov EBP,ESI
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+65536]
  mov ESI,[ESI]Chip.pNextChip

  test DL,DL
  jne NOPn_Special

NOPn_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// separate trap handlers avoid cache contamination

IRQ1_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je IRQ1_Adjust

IRQ1_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp IRQ1_Continue

IRQ1_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  mov [EBP]CPU65xx.pbPCBase,EAX
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae IRQ1_Trap
  jmp IRQ1_Continue

IRQ2_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je IRQ2_Adjust

IRQ2_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp IRQ2_Continue

IRQ2_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  mov [EBP]CPU65xx.pbPCBase,EAX
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae IRQ2_Trap
  jmp IRQ2_Continue

IRQ6_Trap:

  and EDX,0x000000FF
  mov EAX,0xFFFE
  mov [ESI]CPU65xx.iRWAddress,EAX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp IRQ6_Continue

IRQ7_Trap:

  and EDX,0x000000FF
  mov EAX,0xFFFF
  mov [ESI]CPU65xx.iRWAddress,EAX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  mov AL,byte ptr ([ESI]CPU65xx.iAddr16[0])
  jmp IRQ7_Continue

NMI1_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je NMI1_Adjust

NMI1_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp NMI1_Continue

NMI1_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  mov [EBP]CPU65xx.pbPCBase,EAX
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae NMI1_Trap
  jmp NMI1_Continue

NMI2_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je NMI2_Adjust

NMI2_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp NMI2_Continue

NMI2_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  mov [EBP]CPU65xx.pbPCBase,EAX
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae NMI2_Trap
  jmp NMI2_Continue

NMI6_Trap:

  and EDX,0x000000FF
  mov EAX,0xFFFC
  mov [ESI]CPU65xx.iRWAddress,EAX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp NMI6_Continue

NMI7_Trap:

  and EDX,0x000000FF
  mov EAX,0xFFFB
  mov [ESI]CPU65xx.iRWAddress,EAX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  xor EBX,EBX
  mov BH,AL
  mov BL,byte ptr ([ESI]CPU65xx.iAddr16[0])
  mov EAX,EBX
  jmp NMI7_Continue

Command_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [EBP]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  mov EAX,EBX
  je Command_Adjust

Command_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  and EAX,0x000000FF
  jmp Command_Continue

Command_Adjust:

  shr EAX,12
  inc EBX
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCBase,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  mov DL,[EBX+65536-1]
  dec EBX
  cmp DL,2
  jae Command_Trap
  xor EAX,EAX
  mov AL,[EBX]
  jmp Command_Continue

RWz2_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je RWz2_Adjust

RWz2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RWz2_Continue

RWz2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae RWz2_Trap
  mov AL,[EBX]
  jmp RWz2_Continue

RWzx2_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je RWzx2_Adjust

RWzx2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RWzx2_Continue

RWzx2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae RWzx2_Trap
  mov AL,[EBX]
  jmp RWzx2_Continue

RWzx3_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp RWzx3_Continue

RWzy2_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je RWzy2_Adjust

RWzy2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RWzy2_Continue

RWzy2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae RWzy2_Trap
  mov AL,[EBX]
  jmp RWzy2_Continue

RWzy3_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp RWzy3_Continue

RWa2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je RWa2_Adjust

RWa2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RWa2_Continue

RWa2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae RWa2_Trap
  mov AL,[EBX]
  jmp RWa2_Continue

RWa3_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je RWa3_Adjust

RWa3_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RWa3_Continue

RWa3_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae RWa3_Trap
  mov AL,[EBX]
  jmp RWa3_Continue

Rax2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Rax2_Adjust

Rax2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Rax2_Continue

Rax2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Rax2_Trap
  mov AL,[EBX]
  jmp Rax2_Continue

Rax3_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  push EAX
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Rax3_Adjust

Rax3_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov BL,AL
  pop EAX
  mov AH,BL
  jmp Rax3_Continue

Rax3_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Rax3_Trap
  pop EAX
  mov AH,[EBX]
  jmp Rax3_Continue

Raxy4_Trap:

  and EDX,0x000000FF
  mov [EBP]CPU65xx.iRWAddress,ECX
  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Raxy4_Continue

Ray2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Ray2_Adjust

Ray2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Ray2_Continue

Ray2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Ray2_Trap
  mov AL,[EBX]
  jmp Ray2_Continue

Ray3_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  push EAX
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Ray3_Adjust

Ray3_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov BL,AL
  pop EAX
  mov AH,BL
  jmp Ray3_Continue

Ray3_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Ray3_Trap
  pop EAX
  mov AH,[EBX]
  jmp Ray3_Continue

Wax2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Wax2_Adjust

Wax2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Wax2_Continue

Wax2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Wax2_Trap
  mov AL,[EBX]
  jmp Wax2_Continue

Wax3_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  push EAX
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Wax3_Adjust

Wax3_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov BL,AL
  pop EAX
  mov AH,BL
  jmp Wax3_Continue

Wax3_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Wax3_Trap
  pop EAX
  mov AH,[EBX]
  jmp Wax3_Continue

Waxy4_Trap:

  and EDX,0x000000FF
  mov [EBP]CPU65xx.iRWAddress,ECX
  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Waxy4_Continue

Way2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Way2_Adjust

Way2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Way2_Continue

Way2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Way2_Trap
  mov AL,[EBX]
  jmp Way2_Continue

Way3_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  push EAX
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Way3_Adjust

Way3_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov BL,AL
  pop EAX
  mov AH,BL
  jmp Way3_Continue

Way3_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Way3_Trap
  pop EAX
  mov AH,[EBX]
  jmp Way3_Continue

RWix2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je RWix2_Adjust

RWix2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RWix2_Continue

RWix2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae RWix2_Trap
  mov AL,[EBX]
  jmp RWix2_Continue

RWix3_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp RWix3_Continue

RWix4_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RWix4_Continue

RWix5_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RWix5_Continue

Riy2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Riy2_Adjust

Riy2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Riy2_Continue

Riy2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Riy2_Trap
  mov AL,[EBX]
  jmp Riy2_Continue

Riy3_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Riy3_Continue

Riy4_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Riy4_Continue

Wiy2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Wiy2_Adjust

Wiy2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Wiy2_Continue

Wiy2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Wiy2_Trap
  mov AL,[EBX]
  jmp Wiy2_Continue

Wiy3_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Wiy3_Continue

Wiy4_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov BL,AL
  pop EAX
  mov AH,BL
  jmp Wiy4_Continue

Mz2_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Mz2_Adjust

Mz2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Mz2_Continue

Mz2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Mz2_Trap
  mov AL,[EBX]
  jmp Mz2_Continue

Mz3_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Mz3_Continue

Mzx2_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Mzx2_Adjust

Mzx2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Mzx2_Continue

Mzx2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Mzx2_Trap
  mov AL,[EBX]
  jmp Mzx2_Continue

Mzx3_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp Mzx3_Continue

Mzx4_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Mzx4_Continue

Ma2_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Ma2_Adjust

Ma2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Ma2_Continue

Ma2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Ma2_Trap
  mov AL,[EBX]
  jmp Ma2_Continue

Ma3_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Ma3_Adjust

Ma3_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Ma3_Continue

Ma3_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Ma3_Trap
  mov AL,[EBX]
  jmp Ma3_Continue

Ma4_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Ma4_Continue

Max2_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Max2_Adjust

Max2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Max2_Continue

Max2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Max2_Trap
  mov AL,[EBX]
  jmp Max2_Continue

Max3_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  push EAX
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Max3_Adjust

Max3_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov BL,AL
  pop EAX
  mov AH,BL
  jmp Max3_Continue

Max3_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Max3_Trap
  pop EAX
  mov AH,[EBX]
  jmp Max3_Continue

Maxy4_Trap:

  and EDX,0x000000FF
  mov [EBP]CPU65xx.iRWAddress,ECX
  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Maxy4_Continue

BMIr2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je BMIr2_Adjust

BMIr2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[0])
  jmp BMIr2_Continue

BMIr2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae BMIr2_Trap
  mov AL,[EBX]
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[0])
  jmp BMIr2_Continue

BPLr2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je BPLr2_Adjust

BPLr2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[0])
  jmp BPLr2_Continue

BPLr2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae BPLr2_Trap
  mov AL,[EBX]
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[0])
  jmp BPLr2_Continue

BCSr2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je BCSr2_Adjust

BCSr2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  jmp BCSr2_Continue

BCSr2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae BCSr2_Trap
  mov AL,[EBX]
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  jmp BCSr2_Continue

BCCr2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je BCCr2_Adjust

BCCr2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  jmp BCCr2_Continue

BCCr2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae BCCr2_Trap
  mov AL,[EBX]
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  jmp BCCr2_Continue

BVSr2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je BVSr2_Adjust

BVSr2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,[ESI]CPU65xx.bVFlag
  jmp BVSr2_Continue

BVSr2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae BVSr2_Trap
  mov AL,[EBX]
  mov AH,[ESI]CPU65xx.bVFlag
  jmp BVSr2_Continue

BVCr2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je BVCr2_Adjust

BVCr2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,[ESI]CPU65xx.bVFlag
  jmp BVCr2_Continue

BVCr2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae BVCr2_Trap
  mov AL,[EBX]
  mov AH,[ESI]CPU65xx.bVFlag
  jmp BVCr2_Continue

BEQr2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je BEQr2_Adjust

BEQr2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,[ESI]CPU65xx.bZVal
  jmp BEQr2_Continue

BEQr2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae BEQr2_Trap
  mov AL,[EBX]
  mov AH,[ESI]CPU65xx.bZVal
  jmp BEQr2_Continue

BNEr2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je BNEr2_Adjust

BNEr2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,[ESI]CPU65xx.bZVal
  jmp BNEr2_Continue

BNEr2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae BNEr2_Trap
  mov AL,[EBX]
  mov AH,[ESI]CPU65xx.bZVal
  jmp BNEr2_Continue

Br3_Special:

  push EAX
  and EBX,0x0000FFFF
  push EBX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Br3_Adjust

Br3_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov ECX,[ESI]CPU65xx.pbPCBase
  pop EBX
  pop EAX
  jmp Br3_Continue

Br3_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov ECX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  mov [ESI]CPU65xx.pbPCBase,ECX
  mov DL,[EBX+ECX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Br3_Trap
  pop EBX
  pop EAX
  jmp Br3_Continue

Br4_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Br4_Adjust

Br4_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Br4_Continue

Br4_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Br4_Trap
  jmp Br4_Continue

LDAb_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je LDAb_Adjust

LDAb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  jmp LDAb_Continue

LDAb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae LDAb_Trap
  mov AH,[EBX]
  jmp LDAb_Continue

LDA8_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov [ESI]CPU65xx.bAccu,AL
  jmp LDA8_Continue

LDA16_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp LDA16_Continue

LDXb_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je LDXb_Adjust

LDXb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp LDXb_Continue

LDXb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae LDXb_Trap
  mov AL,[EBX]
  jmp LDXb_Continue

LDX8_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov byte ptr ([ESI]CPU65xx.iXReg),AL
  jmp LDX8_Continue

LDX16_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp LDX16_Continue

LDYb_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je LDYb_Adjust

LDYb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp LDYb_Continue

LDYb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae LDYb_Trap
  mov AL,[EBX]
  jmp LDYb_Continue

LDY8_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov byte ptr ([ESI]CPU65xx.iYReg),AL
  jmp LDY8_Continue

LDY16_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp LDY16_Continue

TAXn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  push EAX
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je TAXn_Adjust

TAXn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp TAXn_Continue

TAXn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae TAXn_Trap
  pop EAX
  jmp TAXn_Continue

TAYn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  push EAX
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je TAYn_Adjust

TAYn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp TAYn_Continue

TAYn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae TAYn_Trap
  pop EAX
  jmp TAYn_Continue

TXAn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  push EAX
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je TXAn_Adjust

TXAn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp TXAn_Continue

TXAn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae TXAn_Trap
  pop EAX
  jmp TXAn_Continue

TYAn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  push EAX
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je TYAn_Adjust

TYAn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp TYAn_Continue

TYAn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae TYAn_Trap
  pop EAX
  jmp TYAn_Continue

TSXn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  push EAX
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je TSXn_Adjust

TSXn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp TSXn_Continue

TSXn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae TSXn_Trap
  pop EAX
  jmp TSXn_Continue

TXSn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  push EAX
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je TXSn_Adjust

TXSn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp TXSn_Continue

TXSn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae TXSn_Trap
  pop EAX
  jmp TXSn_Continue

PHx2_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [EBP]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je PHx2_Adjust

PHx2_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp PHx2_Continue

PHx2_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  mov [EBP]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae PHx2_Trap
  jmp PHx2_Continue

PLx2_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je PLx2_Adjust

PLx2_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp PLx2_Continue

PLx2_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  mov [EBP]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae PLx2_Trap
  jmp PLx2_Continue

PLx3_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp PLx3_Continue

PLAn4_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov [ESI]CPU65xx.bAccu,AL
  jmp PLAn4_Continue

PLPn4_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  jmp PLPn4_Continue

INXn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je INXn_Adjust

INXn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp INXn_Continue

INXn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae INXn_Trap
  jmp INXn_Continue

INYn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je INYn_Adjust

INYn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp INYn_Continue

INYn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae INYn_Trap
  jmp INYn_Continue

DEXn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je DEXn_Adjust

DEXn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp DEXn_Continue

DEXn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae DEXn_Trap
  jmp DEXn_Continue

DEYn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je DEYn_Adjust

DEYn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp DEYn_Continue

DEYn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae DEYn_Trap
  jmp DEYn_Continue

ASLn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  push EAX
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je ASLn_Adjust

ASLn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp ASLn_Continue

ASLn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae ASLn_Trap
  pop EAX
  jmp ASLn_Continue

LSRn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  push EAX
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je LSRn_Adjust

LSRn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp LSRn_Continue

LSRn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae LSRn_Trap
  pop EAX
  jmp LSRn_Continue

ROLn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  push EAX
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je ROLn_Adjust

ROLn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp ROLn_Continue

ROLn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae ROLn_Trap
  pop EAX
  jmp ROLn_Continue

RORn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  push EAX
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je RORn_Adjust

RORn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp RORn_Continue

RORn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae RORn_Trap
  pop EAX
  jmp RORn_Continue

ANDb_Special:

  push EAX
  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je ANDb_Adjust

ANDb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp ANDb_Continue

ANDb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae ANDb_Trap
  pop EAX
  mov AL,[EBX]
  jmp ANDb_Continue

AND8_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp AND8_Continue

AND16_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp AND16_Continue

ORAb_Special:

  push EAX
  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je ORAb_Adjust

ORAb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp ORAb_Continue

ORAb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae ORAb_Trap
  pop EAX
  mov AL,[EBX]
  jmp ORAb_Continue

ORA8_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp ORA8_Continue

ORA16_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp ORA16_Continue

EORb_Special:

  push EAX
  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je EORb_Adjust

EORb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp EORb_Continue

EORb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae EORb_Trap
  pop EAX
  mov AL,[EBX]
  jmp EORb_Continue

EOR8_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp EOR8_Continue

EOR16_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp EOR16_Continue

CLCn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je CLCn_Adjust

CLCn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  xor AL,AL
  jmp CLCn_Continue

CLCn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae CLCn_Trap
  xor AL,AL
  jmp CLCn_Continue

SECn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je SECn_Adjust

SECn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AL,1
  jmp SECn_Continue

SECn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae SECn_Trap
  mov AL,1
  jmp SECn_Continue

CLDn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je CLDn_Adjust

CLDn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AL,[ESI]CPU65xx.bFlags
  jmp CLDn_Continue

CLDn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae CLDn_Trap
  mov AL,[ESI]CPU65xx.bFlags
  jmp CLDn_Continue

SEDn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je SEDn_Adjust

SEDn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AL,[ESI]CPU65xx.bFlags
  jmp SEDn_Continue

SEDn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae SEDn_Trap
  mov AL,[ESI]CPU65xx.bFlags
  jmp SEDn_Continue

CLIn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je CLIn_Adjust

CLIn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AL,[ESI]CPU65xx.bFlags
  jmp CLIn_Continue

CLIn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae CLIn_Trap
  mov AL,[ESI]CPU65xx.bFlags
  jmp CLIn_Continue

SEIn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je SEIn_Adjust

SEIn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AL,[ESI]CPU65xx.bFlags
  jmp SEIn_Continue

SEIn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae SEIn_Trap
  mov AL,[ESI]CPU65xx.bFlags
  jmp SEIn_Continue

CLVn_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je CLVn_Adjust

CLVn_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  xor AL,AL
  jmp CLVn_Continue

CLVn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae CLVn_Trap
  xor AL,AL
  jmp CLVn_Continue

ADCb_Special:

  push EAX
  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je ADCb_Adjust

ADCb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp ADCb_Continue

ADCb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae ADCb_Trap
  pop EAX
  mov AL,[EBX]
  jmp ADCb_Continue

ADC8_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp ADC8_Continue

ADC16_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp ADC16_Continue

SBCb_Special:

  push EAX
  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je SBCb_Adjust

SBCb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp SBCb_Continue

SBCb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae SBCb_Trap
  pop EAX
  mov AL,[EBX]
  jmp SBCb_Continue

SBC8_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp SBC8_Continue

SBC16_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp SBC16_Continue

CMPb_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je CMPb_Adjust

CMPb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  mov AL,[ESI]CPU65xx.bAccu
  jmp CMPb_Continue

CMPb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae CMPb_Trap
  mov AH,[EBX]
  mov AL,[ESI]CPU65xx.bAccu
  jmp CMPb_Continue

CMP8_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  mov AL,[ESI]CPU65xx.bAccu
  jmp CMP8_Continue

CMP16_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  mov AL,[ESI]CPU65xx.bAccu
  jmp CMP16_Continue

CPXb_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je CPXb_Adjust

CPXb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)
  jmp CPXb_Continue

CPXb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae CPXb_Trap
  mov AH,[EBX]
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)
  jmp CPXb_Continue

CPX8_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)
  jmp CPX8_Continue

CPX16_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)
  jmp CPX16_Continue

CPYb_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je CPYb_Adjust

CPYb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  mov AL,byte ptr ([ESI]CPU65xx.iYReg)
  jmp CPYb_Continue

CPYb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae CPYb_Trap
  mov AH,[EBX]
  mov AL,byte ptr ([ESI]CPU65xx.iYReg)
  jmp CPYb_Continue

CPY8_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  mov AL,byte ptr ([ESI]CPU65xx.iYReg)
  jmp CPY8_Continue

CPY16_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  mov AL,byte ptr ([ESI]CPU65xx.iYReg)
  jmp CPY16_Continue

BIT8_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,[ESI]CPU65xx.bAccu
  jmp BIT8_Continue

BIT16_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,[ESI]CPU65xx.bAccu
  jmp BIT16_Continue

BRKn2_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je BRKn2_Adjust

BRKn2_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp BRKn2_Continue

BRKn2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [EBP]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae BRKn2_Trap
  jmp BRKn2_Continue

BRKn6_Trap:

  and EDX,0x000000FF
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp BRKn6_Continue

BRKn7_Trap:

  and EDX,0x000000FF
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  mov AL,byte ptr ([ESI]CPU65xx.iAddr16[0])
  jmp BRKn7_Continue

RTIn2_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je RTIn2_Adjust

RTIn2_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RTIn2_Continue

RTIn2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [EBP]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae RTIn2_Trap
  jmp RTIn2_Continue

RTIn3_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RTIn3_Continue

RTIn4_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  jmp RTIn4_Continue

RTIn5_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RTIn5_Continue

RTIn6_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),AL
  jmp RTIn6_Continue

JSRw2_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je JSRw2_Adjust

JSRw2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp JSRw2_Continue

JSRw2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae JSRw2_Trap
  mov AL,[EBX]
  jmp JSRw2_Continue

JSRw3_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp JSRw3_Continue

JSRw6_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je JSRw6_Adjust

JSRw6_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov BL,AL
  mov EAX,[ESI]CPU65xx.iAddr16
  mov AH,BL
  jmp JSRw6_Continue

JSRw6_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae JSRw6_Trap
  mov EAX,[ESI]CPU65xx.iAddr16
  mov AH,[EBX]
  jmp JSRw6_Continue

RTSn2_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je RTSn2_Adjust

RTSn2_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RTSn2_Continue

RTSn2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [EBP]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae RTSn2_Trap
  jmp RTSn2_Continue

RTSn3_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RTSn3_Continue

RTSn4_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RTSn4_Continue

RTSn5_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),AL
  jmp RTSn5_Continue

RTSn6_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je RTSn6_Adjust

RTSn6_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp RTSn6_Continue

RTSn6_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [EBP]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae RTSn6_Trap
  jmp RTSn6_Continue

JMPw2_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je JMPw2_Adjust

JMPw2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp JMPw2_Continue

JMPw2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae JMPw2_Trap
  mov AL,[EBX]
  jmp JMPw2_Continue

JMPw3_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je JMPw3_Adjust

JMPw3_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov BL,AL
  mov EAX,[ESI]CPU65xx.iAddr16
  mov AH,BL
  jmp JMPw3_Continue

JMPw3_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae JMPw3_Trap
  mov EAX,[ESI]CPU65xx.iAddr16
  mov AH,[EBX]
  jmp JMPw3_Continue

JMPi2_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je JMPi2_Adjust

JMPi2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp JMPi2_Continue

JMPi2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae JMPi2_Trap
  mov AL,[EBX]
  jmp JMPi2_Continue

JMPi3_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je JMPi3_Adjust

JMPi3_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp JMPi3_Continue

JMPi3_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae JMPi3_Trap
  mov AL,[EBX]
  jmp JMPi3_Continue

JMPi4_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp JMPi4_Continue

JMPi5_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov BL,AL
  mov EAX,[ESI]CPU65xx.iAddr8
  mov AH,BL
  jmp JMPi5_Continue

NOPn_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je NOPn_Adjust

NOPn_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp NOPn_Continue

NOPn_Adjust:

  mov EAX,EBX
  shr EAX,12
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  mov [EBP]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iDummyStart
  jae NOPn_Trap
  jmp NOPn_Continue


////////////////////////////////////////////////////////////////////////////////
// undocumented/illegal commands


////////////////////////////////////////////////////////////////////////////////
// May - Modify command, absolute+y addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch address ll, PC += 1                       May2
//  3    PC    R  fetch address hh, PC += 1, ll += yreg           May3
//  4   hhll   R  dummy read, hh += carry(ll)                     Maxy4
//  5   hhll   R  read data                                       Ma4
//  6   hhll   W  write old data, perform action                  pfnAction
//  7   hhll   W  write new data                                  Ma6

proc(May2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset May3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne May2_Special

May2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(May3) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  xor ECX,ECX

  mov EAX,[ESI]CPU65xx.iYReg
  mov CL,byte ptr ([ESI]CPU65xx.iAddr16[0])

  add EAX,ECX
  mov EDX,offset Maxy4+4

  lea ECX,[EBX+1]
  mov [ESI]Chip.pfnOnClock,EDX

  mov [ESI]CPU65xx.pbPCPtr,ECX
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bFixAdr,AH
  xor EDX,EDX

  mov AH,[EBX]
  mov DL,[EBX+65536]

  test DL,DL
  jne May3_Special

May3_Continue:

  mov [ESI]CPU65xx.iAddr16,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Mix - Modify command, indirect+x addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch pointer pp, PC += 1                       Mix2
//  3   00pp   R  dummy read, pp += xreg                          Mix3
//  4   00pp   R  read address ll, pp += 1                        Mix4
//  5   00pp   R  read address hh                                 Mix5
//  6   hhll   R  read data                                       Ma4
//  7   hhll   W  write old data, perform action                  pfnAction
//  8   hhll   W  write new data                                  Ma6

proc(Mix2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Mix3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Mix2_Special

Mix2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Mix3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,offset Mix4+4
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)

  mov [ESI]Chip.pfnOnClock,EDX
  add AL,CL

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iDummyStart
  jae Mix3_Trap

Mix3_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Mix4) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  xor EDX,EDX
  mov EBP,offset Mix5+4

  mov DL,[EBX+ECX+65536]
  lea EAX,[ECX+1]

  mov [ESI]Chip.pfnOnClock,EBP
  mov byte ptr ([ESI]CPU65xx.iAddr8),AL

  mov EBP,[ESI]Chip.pNextChip
  mov AL,[EBX+ECX]

  cmp DL,CPU16x4k_iIOStart
  jae Mix4_Trap

Mix4_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Mix5) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  xor EDX,EDX
  mov EBP,offset Ma4+4

  mov AL,[EBX+ECX]
  mov [ESI]Chip.pfnOnClock,EBP

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae Mix5_Trap

Mix5_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// Miy - Modify command, indirect+y addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch pointer pp, PC += 1                       Miy2
//  3   00pp   R  read address ll, pp += 1                        Miy3
//  4   00pp   R  read address hh, ll += yreg                     Miy4
//  5   hhll   R  dummy read, hh += carry(ll)                     Miy5
//  6   hhll   R  read data                                       Ma4
//  7   hhll   W  write old data, perform action                  pfnAction
//  8   hhll   W  write new data                                  Ma6

proc(Miy2) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,offset Miy3+4

  mov EBP,[ESI]Chip.pNextChip
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne Miy2_Special

Miy2_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Miy3) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EBP,offset Miy4+4
  xor EDX,EDX

  mov [ESI]Chip.pfnOnClock,EBP
  lea EAX,[ECX+1]

  mov byte ptr ([ESI]CPU65xx.iAddr8),AL
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae Miy3_Trap

Miy3_Continue:

  mov byte ptr ([ESI]CPU65xx.iAddr16[0]),AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Miy4) index(0)

  xor EAX,EAX
  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]

  mov AL,byte ptr ([ESI]CPU65xx.iAddr16)
  mov EDX,[ESI]CPU65xx.iYReg

  add EAX,EDX
  mov ECX,[ESI]CPU65xx.iAddr8

  mov [ESI]CPU65xx.bFixAdr,AH
  mov EBP,offset Miy5+4

  mov AH,[EBX+ECX]
  mov [ESI]Chip.pfnOnClock,EBP

  mov DL,[EBX+ECX+65536]
  mov EBP,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae Miy4_Trap

Miy4_Continue:

  mov [ESI]CPU65xx.iAddr16,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp

proc(Miy5) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,offset Ma4+4

  shr EAX,12
  mov ECX,[ESI]CPU65xx.iAddr16

  mov [ESI]Chip.pfnOnClock,EDX
  mov DL,[ESI]CPU65xx.bFixAdr

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add DL,CH

  mov byte ptr ([ESI]CPU65xx.iAddr16[1]),DL
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iDummyStart
  jae Miy5_Trap

Miy5_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// NOPb - No operation (no flags), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, perform action                      NOPb

proc(NOPb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt
  
  mov EBP,ESI
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+65536]
  lea ECX,[EBX+1]

  mov ESI,[ESI]Chip.pNextChip
  mov [EBP]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne NOPb_Special

NOPb_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// NOP8 - No operation (no flags), z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   R  dummy read, perform action                      NOP8

proc(NOP8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iDummyStart
  jae NOP8_Trap

NOP8_Continue:

  mov ESI,EBP
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// NOP16 - No operation (no flags), a/ax addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  dummy read, perform action                      NOP16

proc(NOP16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov ECX,[ESI]CPU65xx.iAddr16
  mov [ESI]Chip.pfnOnClock,EDX

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  mov EBP,ESI

  mov ESI,[ESI]Chip.pNextChip
  nop

  mov DL,[EBX+ECX+65536]
  nop

  cmp DL,CPU16x4k_iDummyStart
  jae NOP16_Trap

NOP16_Continue:

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ASO8 - Arithmetic shift left memory (C) and OR accumulator (NZ),
// z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-5  00ll   W  write old data, perform action                  ASO8

proc(ASO8) index(0)

  mov DL,[ESI]CPU65xx.bData
  mov AH,[ESI]CPU65xx.bAccu

  shl EDX,1
  mov AL,[ESI]CPU65xx.bData

  mov [ESI]CPU65xx.bData,DL
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  or DL,AH
  mov ECX,[ESI]CPU65xx.iAddr8

  mov [ESI]CPU65xx.bAccu,DL
  mov EDI,offset Mz5+4

  mov [ESI]CPU65xx.bZVal,DL
  mov EBP,ESI

  mov [ESI]CPU65xx.iNCFlag,EDX
  mov [ESI]Chip.pfnOnClock,EDI

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ASO16 - Arithmetic shift left memory (C) and OR accumulator (NZ),
// a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 5-6  hhll   W  write old data, perform action                  ASO16

proc(ASO16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov DL,[ESI]CPU65xx.bData

  shr EAX,12
  mov ECX,[ESI]CPU65xx.iAddr16

  shl EDX,1
  mov EBP,offset Ma6+4

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EAX*4]
  mov AL,[ESI]CPU65xx.bData

  mov [ESI]CPU65xx.bData,DL
  mov AH,[ESI]CPU65xx.bAccu

  or DL,AH
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.bAccu,DL
  mov EBP,ESI

  mov [ESI]CPU65xx.bZVal,DL
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.iNCFlag,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// RLA8 - Rotate left memory (C) and AND accumulator (NZ),
// z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-5  00ll   W  write old data, perform action                  RLA8

proc(RLA8) index(0)

  mov BL,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov DL,[ESI]CPU65xx.bData

  shr BL,1
  mov AH,[ESI]CPU65xx.bAccu

  adc EDX,EDX
  mov AL,[ESI]CPU65xx.bData

  mov [ESI]CPU65xx.bData,DL
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  and DL,AH
  mov ECX,[ESI]CPU65xx.iAddr8

  mov [ESI]CPU65xx.bAccu,DL
  mov EDI,offset Mz5+4

  mov [ESI]CPU65xx.bZVal,DL
  mov EBP,ESI

  mov [ESI]CPU65xx.iNCFlag,EDX
  mov [ESI]Chip.pfnOnClock,EDI

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// RLA16 - Rotate left memory (C) and AND accumulator (NZ),
// a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 5-6  hhll   W  write old data, perform action                  RLA16

proc(RLA16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov DL,[ESI]CPU65xx.bData

  shr EAX,12
  mov BL,byte ptr ([ESI]CPU65xx.iNCFlag[1])

  shr BL,1
  mov ECX,[ESI]CPU65xx.iAddr16

  adc EDX,EDX
  mov EBP,offset Ma6+4

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EAX*4]
  mov AL,[ESI]CPU65xx.bData

  mov [ESI]CPU65xx.bData,DL
  mov AH,[ESI]CPU65xx.bAccu

  and DL,AH
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.bAccu,DL
  mov EBP,ESI

  mov [ESI]CPU65xx.bZVal,DL
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.iNCFlag,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LSE8 - Logical shift right memory (C) and EOR accumulator (NZ),
// z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-5  00ll   W  write old data, perform action                  LSE8

proc(LSE8) index(0)

  mov DL,[ESI]CPU65xx.bData
  mov AH,[ESI]CPU65xx.bAccu

  shr DL,1
  mov AL,[ESI]CPU65xx.bData

  sbb DH,DH
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov [ESI]CPU65xx.bData,DL
  xor DL,AH

  mov [ESI]CPU65xx.bAccu,DL
  mov ECX,[ESI]CPU65xx.iAddr8

  mov [ESI]CPU65xx.bZVal,DL
  mov EBP,offset Mz5+4

  mov [ESI]CPU65xx.iNCFlag,EDX
  mov [ESI]Chip.pfnOnClock,EBP

  mov [EBX+ECX],AL
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LSE16 - Logical shift right memory (C) and EOR accumulator (NZ),
// a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 5-6  hhll   W  write old data, perform action                  LSE16

proc(LSE16) index(0)

  mov EBX,[ESI]CPU65xx.iAddr16
  mov DL,[ESI]CPU65xx.bData

  shr EBX,12
  mov ECX,[ESI]CPU65xx.iAddr16

  shr DL,1
  mov EBP,offset Ma6+4

  sbb DH,DH
  mov AL,[ESI]CPU65xx.bData

  mov AH,[ESI]CPU65xx.bAccu
  mov [ESI]CPU65xx.bData,DL

  xor DL,AH
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EBX*4]

  mov [ESI]CPU65xx.bAccu,DL
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.bZVal,DL
  mov EBP,ESI

  mov [EBX+ECX],AL
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.iNCFlag,EDX
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// RRA8 - Rotate right memory (C) and ADC accumulator (NVZC),
// z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-5  00ll   W  write old data, perform action                  RRA8

proc(RRA8) index(0)

  mov AL,[ESI]CPU65xx.bFlags
  mov DL,[ESI]CPU65xx.bData

  test AL,0x08
  jne RRA8_Dec

  mov DH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov AL,[ESI]CPU65xx.bData

  shr EDX,1
  mov AH,[ESI]CPU65xx.bAccu

  mov [ESI]CPU65xx.bData,DL
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  adc DL,AH
  mov ECX,[ESI]CPU65xx.iAddr8

  ;two
  ;clocks

  ;unpaired
  seto AH

  sbb DH,DH
  mov [ESI]CPU65xx.bVFlag,AH

  mov [ESI]CPU65xx.bAccu,DL
  mov EBP,offset Mz5+4

  mov [ESI]CPU65xx.bZVal,DL
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.iNCFlag,EDX
  mov EBP,ESI

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

RRA8_Dec:

  push EDX
  mov AL,DL

  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov EBX,offset Mz5+4

  shr EAX,1
  mov [ESI]Chip.pfnOnClock,EBX

  sbb EBP,EBP
  mov [ESI]CPU65xx.bData,AL

  shr EBP,1
  mov AH,[ESI]CPU65xx.bAccu

  mov ECX,EAX
  mov EBX,EAX

  adc CH,CL
  and EAX,0x0F0F

  shr EBP,1
  mov [ESI]CPU65xx.bZVal,CH

  adc AH,AL
  and EBX,0xF0F0

  cmp AH,0x0A
  jb RRA8_LowOK

  sub AH,0x0A
  or EBX,0x0808

RRA8_LowOK:

  add BH,BL
  mov CL,1

  ;two
  ;clocks

  ;unpaired
  seto DH

  sbb DL,DL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),BH

  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),DL
  jne RRA8_AdjustHigh

  cmp BH,0xA0
  jb RRA8_HighOK

RRA8_AdjustHigh:

  sub BH,0xA0
  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),CL

RRA8_HighOK:

  and AH,0x0F
  mov [ESI]CPU65xx.bVFlag,DH

  or AH,BH
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov [ESI]CPU65xx.bAccu,AH
  mov ECX,[ESI]CPU65xx.iAddr8

  pop EAX
  mov EBP,ESI

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// RRA16 - Rotate right memory (C) and ADC accumulator (NVZC),
// a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 5-6  hhll   W  write old data, perform action                  RRA16

proc(RRA16) index(0)

  mov AL,[ESI]CPU65xx.bFlags
  mov DL,[ESI]CPU65xx.bData

  test AL,0x08
  jne RRA16_Dec

  mov EBX,[ESI]CPU65xx.iAddr16
  mov DH,byte ptr ([ESI]CPU65xx.iNCFlag[1])

  shr EBX,12
  mov ECX,[ESI]CPU65xx.iAddr16

  shr EDX,1
  mov AL,[ESI]CPU65xx.bData

  mov [ESI]CPU65xx.bData,DL
  mov AH,[ESI]CPU65xx.bAccu

  adc DL,AH
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EBX*4]

  ;two
  ;clocks

  ;unpaired
  seto AH

  sbb DH,DH
  mov [ESI]CPU65xx.bVFlag,AH

  mov [ESI]CPU65xx.bAccu,DL
  mov EBP,offset Ma6+4

  mov [ESI]CPU65xx.bZVal,DL
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.iNCFlag,EDX
  mov EBP,ESI

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

RRA16_Dec:

  push EDX
  mov AL,DL

  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov EBX,offset Ma6+4

  shr EAX,1
  mov [ESI]Chip.pfnOnClock,EBX

  sbb EBP,EBP
  mov [ESI]CPU65xx.bData,AL

  shr EBP,1
  mov AH,[ESI]CPU65xx.bAccu

  mov ECX,EAX
  mov EBX,EAX

  adc CH,CL
  and EAX,0x0F0F

  shr EBP,1
  mov [ESI]CPU65xx.bZVal,CH

  adc AH,AL
  and EBX,0xF0F0

  cmp AH,0x0A
  jb RRA16_LowOK

  sub AH,0x0A
  or EBX,0x0808

RRA16_LowOK:

  add BH,BL
  mov CL,1

  ;two
  ;clocks

  ;unpaired
  seto DH

  sbb DL,DL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),BH

  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),DL
  jne RRA16_AdjustHigh

  cmp BH,0xA0
  jb RRA16_HighOK

RRA16_AdjustHigh:

  sub BH,0xA0
  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),CL

RRA16_HighOK:

  mov EBP,[ESI]CPU65xx.iAddr16
  and AH,0x0F

  shr EBP,12
  mov [ESI]CPU65xx.bVFlag,DH

  mov ECX,[ESI]CPU65xx.iAddr16
  or AH,BH

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EBP*4]
  mov [ESI]CPU65xx.bAccu,AH

  pop EAX
  mov EBP,ESI

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// DCM8 - Decrement memory (no flags) and CMP accumulator (NZC),
// z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-5  00ll   W  write old data, perform action                  DCM8

proc(DCM8) index(0)

  mov CL,[ESI]CPU65xx.bData
  mov DL,[ESI]CPU65xx.bAccu

  dec ECX
  mov AL,[ESI]CPU65xx.bData

  sub DL,CL
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  sbb DH,DH
  mov [ESI]CPU65xx.bData,CL

  not DH
  mov EBP,offset Mz5+4

  mov [ESI]CPU65xx.bZVal,DL
  mov ECX,[ESI]CPU65xx.iAddr8

  mov [ESI]CPU65xx.iNCFlag,EDX
  mov [ESI]Chip.pfnOnClock,EBP

  mov EBP,ESI
  mov [EBX+ECX],AL

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// DCM16 - Decrement memory (no flags) and CMP accumulator (NZC),
// a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 5-6  hhll   W  write old data, perform action                  DCM16

proc(DCM16) index(0)

  mov EBX,[ESI]CPU65xx.iAddr16
  mov CL,[ESI]CPU65xx.bData

  shr EBX,12
  mov DL,[ESI]CPU65xx.bAccu

  dec ECX
  mov AL,[ESI]CPU65xx.bData

  sub DL,CL
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EBX*4]

  sbb DH,DH
  mov [ESI]CPU65xx.bData,CL

  not DH
  mov EBP,offset Ma6+4

  mov [ESI]CPU65xx.bZVal,DL
  mov ECX,[ESI]CPU65xx.iAddr16

  mov [ESI]CPU65xx.iNCFlag,EDX
  mov [ESI]Chip.pfnOnClock,EBP

  mov [EBX+ECX],AL
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// INS8 - Increment memory (no flags) and SBC accumulator (NVZC),
// z/zx addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-5  00ll   W  write old data, perform action                  INS8

proc(INS8) index(0)

  mov AL,[ESI]CPU65xx.bFlags
  mov CL,[ESI]CPU65xx.bData

  test AL,0x08
  jne INS8_Dec

  mov AL,[ESI]CPU65xx.bData
  inc ECX

  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov [ESI]CPU65xx.bData,CL

  not AH
  mov DL,[ESI]CPU65xx.bAccu

  shr AH,1
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  sbb DL,CL
  mov EBP,offset Mz5+4

  ;two
  ;clocks

  ;unpaired
  seto AH

  sbb DH,DH
  mov [ESI]CPU65xx.bVFlag,AH

  not DH
  mov [ESI]Chip.pfnOnClock,EBP

  mov [ESI]CPU65xx.bAccu,DL
  mov ECX,[ESI]CPU65xx.iAddr8

  mov [ESI]CPU65xx.bZVal,DL
  mov EBP,ESI

  mov [ESI]CPU65xx.iNCFlag,EDX
  mov [EBX+ECX],AL

  mov ESI,[ESI]Chip.pNextChip
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

INS8_Dec:

  lea EAX,[ECX+1]
  mov EBP,offset Mz5+4

  mov [ESI]CPU65xx.bData,AL
  mov [ESI]Chip.pfnOnClock,EBP

  mov CH,byte ptr ([ESI]CPU65xx.iNCFlag)[1]
  mov AH,[ESI]CPU65xx.bAccu

  not CH
  mov EDX,EAX

  shr CH,1
  mov EBX,EAX

  sbb DH,DL
  mov CH,byte ptr ([ESI]CPU65xx.iNCFlag[1])

  not CH
  and EAX,0x0F0F

  and EBX,0xF0F0
  push ECX

  shr CH,1
  mov [ESI]CPU65xx.bZVal,DH

  sbb AH,AL
  jnc INS8_LowOK

  sub AH,0x06
  or BL,0x08

INS8_LowOK:

  sub BH,BL
  mov DL,0

  ;two
  ;clocks

  ;unpaired
  seto DH

  sbb CL,CL
  mov [ESI]CPU65xx.bVFlag,DH

  not CL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),BH

  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),CL
  je INS8_HighOK

  sub BH,0x60
  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),DL

INS8_HighOK:

  and BH,0xF0
  and AH,0x0F

  or AH,BH
  mov ECX,[ESI]CPU65xx.iAddr8

  mov [ESI]CPU65xx.bAccu,AH
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  pop EAX
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// INS16 - Increment memory (no flags) and SBC accumulator (NVZC),
// a/ax/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 5-6  hhll   W  write old data, perform action                  INS16

proc(INS16) index(0)

  mov AL,[ESI]CPU65xx.bFlags
  mov CL,[ESI]CPU65xx.bData

  test AL,0x08
  jne INS16_Dec

  mov AL,[ESI]CPU65xx.bData
  inc ECX

  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])
  mov [ESI]CPU65xx.bData,CL

  not AH
  mov DL,[ESI]CPU65xx.bAccu

  shr AH,1
  mov EBX,[ESI]CPU65xx.iAddr16

  sbb DL,CL
  mov EBP,offset Ma6+4

  ;two
  ;clocks

  ;unpaired
  seto AH

  sbb DH,DH
  mov [ESI]CPU65xx.bVFlag,AH

  shr EBX,12
  not DH

  mov [ESI]CPU65xx.bAccu,DL
  mov ECX,[ESI]CPU65xx.iAddr16

  mov [ESI]CPU65xx.bZVal,DL
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EBX*4]

  mov [ESI]CPU65xx.iNCFlag,EDX
  mov [ESI]Chip.pfnOnClock,EBP

  mov [EBX+ECX],AL
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

INS16_Dec:

  lea EAX,[ECX+1]
  mov EBP,offset Ma6+4

  mov [ESI]CPU65xx.bData,AL
  mov [ESI]Chip.pfnOnClock,EBP

  mov CH,byte ptr ([ESI]CPU65xx.iNCFlag)[1]
  mov AH,[ESI]CPU65xx.bAccu

  not CH
  mov EDX,EAX

  shr CH,1
  mov EBX,EAX

  sbb DH,DL
  mov CH,byte ptr ([ESI]CPU65xx.iNCFlag[1])

  not CH
  and EAX,0x0F0F

  and EBX,0xF0F0
  push ECX

  shr CH,1
  mov [ESI]CPU65xx.bZVal,DH

  sbb AH,AL
  jnc INS16_LowOK

  sub AH,0x06
  or BL,0x08

INS16_LowOK:

  sub BH,BL
  mov DL,0

  ;two
  ;clocks

  ;unpaired
  seto DH

  sbb CL,CL
  mov [ESI]CPU65xx.bVFlag,DH

  not CL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),BH

  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),CL
  je INS16_HighOK

  sub BH,0x60
  mov byte ptr ([ESI]CPU65xx.iNCFlag[1]),DL

INS16_HighOK:

  mov EBP,[ESI]CPU65xx.iAddr16
  and AH,0x0F

  shr EBP,12
  and BH,0xF0

  mov ECX,[ESI]CPU65xx.iAddr16
  or AH,BH

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EBP*4]
  mov [ESI]CPU65xx.bAccu,AH

  pop EAX
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LAX8 - Load accumulator and x register (NZ), z/zy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   R  read data, perform action                       LAX8

proc(LAX8) index(0)

  mov EBX,[ESI]CPU16x4k.apbReadMapping[0*4]
  mov ECX,[ESI]CPU65xx.iAddr8

  mov EDX,[ESI]CPU65xx.pfnInterrupt
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]Chip.pfnOnClock,EDX
  mov AL,[EBX+ECX]

  mov DL,[EBX+ECX+65536]
  mov [ESI]CPU65xx.bAccu,AL

  cmp DL,CPU16x4k_iIOStart
  jae LAX8_Trap

LAX8_Continue:

  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL
  mov byte ptr ([ESI]CPU65xx.iXReg),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LAX16 - Load accumulator and x register (NZ), a/ay/ix/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       LAX16

proc(LAX16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  mov EBP,ESI

  xor EDX,EDX
  mov ESI,[ESI]Chip.pNextChip

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae LAX16_Trap

LAX16_Continue:

  mov [EBP]CPU65xx.bAccu,AL
  mov byte ptr ([EBP]CPU65xx.iXReg),AL

  mov byte ptr ([EBP]CPU65xx.iNCFlag[0]),AL
  mov [EBP]CPU65xx.bZVal,AL

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp



////////////////////////////////////////////////////////////////////////////////
// AXS8 - Store accumulator AND x register (no flags), z/zy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 3-4  00ll   W  perform action, write data                      AXS8

proc(AXS8) index(0)

  mov AL,[ESI]CPU65xx.bAccu
  mov EBX,[ESI]CPU16x4k.apbWriteMapping[0*4]

  mov AH,byte ptr ([ESI]CPU65xx.iXReg)
  mov ECX,[ESI]CPU65xx.iAddr8

  and AL,AH
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov EBP,ESI
  mov [ESI]Chip.pfnOnClock,EDX

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// AXS16 - Store accumulator AND x register (no flags), a/ix addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4/6  hhll   W  perform action, write data                      AXS16

proc(AXS16) index(0)

  mov EBX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EBX,12
  mov AL,[ESI]CPU65xx.bAccu

  mov [ESI]Chip.pfnOnClock,EDX
  mov AH,byte ptr ([ESI]CPU65xx.iXReg)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EBX*4]
  mov ECX,[ESI]CPU65xx.iAddr16

  and AL,AH
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ALRb - AND and LSR (NZC), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             ALRb

proc(ALRb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AH,[ESI]CPU65xx.bAccu
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne ALRb_Special

ALRb_Continue:

  and AL,AH
  mov EBP,ESI

  shr AL,1
  mov ESI,[ESI]Chip.pNextChip

  sbb AH,AH
  ;nop

  mov [EBP]CPU65xx.bAccu,AL
  ;nop

  mov [EBP]CPU65xx.iNCFlag,EAX
  mov [EBP]CPU65xx.bZVal,AL

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ARRb - AND and ROR (NVZC), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             ARRb

proc(ARRb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AH,[ESI]CPU65xx.bAccu
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne ARRb_Special

ARRb_Continue:

  and AL,AH
  mov EBP,[ESI]Chip.pNextChip

  mov BL,AL
  mov CL,AL

  shl EBX,1
  mov AH,byte ptr ([ESI]CPU65xx.iNCFlag[1])

  xor BL,AL
  mov DL,[ESI]CPU65xx.bFlags

  ;unpaired
  sets BL

  shr EAX,1
  mov [ESI]CPU65xx.bVFlag,BL

  mov BL,AL
  and DL,0x08

  mov [ESI]CPU65xx.bZVal,AL
  jne ARRb_Dec

  mov [ESI]CPU65xx.bAccu,AL
  mov [ESI]CPU65xx.iNCFlag,EBX

  ;nop
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

ARRb_Dec:

  mov CH,CL
  xor BH,BH

  and CL,0x0F
  mov AH,AL

  cmp CL,0x05
  jb ARRb_NoAdjustLow

  add AL,0x06
  ;nop

  and EAX,0xF00F
  ;nop

  or AL,AH
  ;nop

ARRb_NoAdjustLow:

  cmp CH,0x50
  jb ARRb_NoAdjustHigh

  add AL,0x60
  not BH

ARRb_NoAdjustHigh:

  mov [ESI]CPU65xx.bAccu,AL
  mov [ESI]CPU65xx.iNCFlag,EBX

  ;nop
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ANEb - A = (A | #$EE) & X & #byte (NZ), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             ANEb

proc(ANEb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AH,[ESI]CPU65xx.bAccu
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne ANEb_Special

ANEb_Continue:

  or AH,0xEE
  mov DL,byte ptr ([ESI]CPU65xx.iXReg)

  and AL,DL
  mov EBP,[ESI]Chip.pNextChip

  and AL,AH
  ;nop

  mov [ESI]CPU65xx.bAccu,AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LXAb - A = X = (A | $EE) & #byte (NZ), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             LXAb

proc(LXAb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AH,[ESI]CPU65xx.bAccu
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne LXAb_Special

LXAb_Continue:

  or AH,0xEE
  mov EBP,ESI

  and AL,AH
  mov ESI,[ESI]Chip.pNextChip

  mov [EBP]CPU65xx.bAccu,AL
  mov byte ptr ([EBP]CPU65xx.iXReg[0]),AL

  mov byte ptr ([EBP]CPU65xx.iNCFlag[0]),AL
  mov [EBP]CPU65xx.bZVal,AL

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// SBXb - X = (A & X) - #byte (NZC), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             SBXb

proc(SBXb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov AL,[ESI]CPU65xx.bAccu

  mov CL,byte ptr ([ESI]CPU65xx.iXReg)
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  and AL,CL
  mov [ESI]Chip.pfnOnClock,EDX

  mov AH,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne SBXb_Special

SBXb_Continue:

  sub AL,AH
  ;nop

  sbb AH,AH
  ;nop

  not AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bZVal,AL
  mov byte ptr ([ESI]CPU65xx.iXReg),AL

  mov [ESI]CPU65xx.iNCFlag,EAX
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// SHA16 - mem = A & X & (hibyte mem + 1) (no flags), ay/iy addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   W  perform action, write data                      SHA16

proc(SHA16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov DL,byte ptr ([ESI]CPU65xx.iYReg)

  sub AL,DL
  mov CL,byte ptr ([ESI]CPU65xx.iXReg)

  sbb AH,0xFF
  mov EBX,[ESI]CPU65xx.iAddr16

  shr EBX,12
  and AH,CL

  mov AL,[ESI]CPU65xx.bAccu
  mov EBP,[ESI]CPU65xx.pfnInterrupt

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EBX*4]
  mov ECX,[ESI]CPU65xx.iAddr16

  and AL,AH
  mov [ESI]Chip.pfnOnClock,EBP

  mov [EBX+ECX],AL
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// SHX16 - mem = X & (hibyte mem + 1) (no flags), ay addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  6   hhll   W  perform action, write data                      SHX16

proc(SHX16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov DL,byte ptr ([ESI]CPU65xx.iYReg)

  sub AL,DL
  mov EBX,[ESI]CPU65xx.iAddr16

  sbb AH,0xFF
  mov EBP,[ESI]CPU65xx.pfnInterrupt

  shr EBX,12
  mov AL,byte ptr ([ESI]CPU65xx.iXReg)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EBX*4]
  mov ECX,[ESI]CPU65xx.iAddr16

  and AL,AH
  mov [ESI]Chip.pfnOnClock,EBP

  mov [EBX+ECX],AL
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// SHY16 - mem = Y & (hibyte mem + 1) (no flags), ax addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  6   hhll   W  perform action, write data                      SHY16

proc(SHY16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov DL,byte ptr ([ESI]CPU65xx.iXReg)

  sub AL,DL
  mov EBX,[ESI]CPU65xx.iAddr16

  sbb AH,0xFF
  mov EBP,[ESI]CPU65xx.pfnInterrupt

  shr EBX,12
  mov AL,byte ptr ([ESI]CPU65xx.iYReg)

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EBX*4]
  mov ECX,[ESI]CPU65xx.iAddr16

  and AL,AH
  mov [ESI]Chip.pfnOnClock,EBP

  mov [EBX+ECX],AL
  mov EBP,ESI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// SHS16 - mem = (S = (A & X)) & (hibyte mem + 1) (no flags), ay addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  6   hhll   W  perform action, write data                      SHS16

proc(SHS16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov DL,byte ptr ([ESI]CPU65xx.iYReg)

  sub AL,DL
  mov CL,byte ptr ([ESI]CPU65xx.iXReg)

  sbb AH,0xFF
  mov EBX,[ESI]CPU65xx.iAddr16

  shr EBX,12
  mov AL,[ESI]CPU65xx.bAccu

  and AL,CL
  mov EBP,ESI

  mov byte ptr ([ESI]CPU65xx.iSP),AL
  mov EDI,[ESI]CPU65xx.pfnInterrupt

  mov EBX,[ESI]CPU16x4k.apbWriteMapping[EBX*4]
  mov ECX,[ESI]CPU65xx.iAddr16

  and AL,AH
  mov [ESI]Chip.pfnOnClock,EDI

  mov DL,[EBX+ECX+65536]
  mov ESI,[ESI]Chip.pNextChip

  cmp DL,CPU16x4k_iIOStart
  jae WriteReg

  mov [EBX+ECX],AL
  ;nop

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// ANCb - AND (NZC), byte addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  1    PC    R  fetch opcode, PC += 1                           Command
//  2    PC    R  fetch data, PC += 1, perform action             ANCb

proc(ANCb) index(0)

  mov EBX,[ESI]CPU65xx.pbPCPtr
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  mov AH,[ESI]CPU65xx.bAccu
  mov [ESI]Chip.pfnOnClock,EDX

  mov AL,[EBX]
  lea ECX,[EBX+1]

  mov DL,[EBX+65536]
  mov [ESI]CPU65xx.pbPCPtr,ECX

  test DL,DL
  jne ANCb_Special

ANCb_Continue:

  and AL,AH
  mov EBP,[ESI]Chip.pNextChip

  ;unpaired
  sets AH

  mov [ESI]CPU65xx.bAccu,AL
  mov [ESI]CPU65xx.iNCFlag,EAX

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// LAS16 - A = X = S = S & data (NZ), ay addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
// 4-6  hhll   R  read data, perform action                       LAS16

proc(LAS16) index(0)

  mov EAX,[ESI]CPU65xx.iAddr16
  mov EDX,[ESI]CPU65xx.pfnInterrupt

  shr EAX,12
  nop

  mov [ESI]Chip.pfnOnClock,EDX
  mov ECX,[ESI]CPU65xx.iAddr16

  mov EBX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  mov AH,byte ptr ([ESI]CPU65xx.iSP)

  ;nop
  ;nop

  mov AL,[EBX+ECX]
  mov DL,[EBX+ECX+65536]

  cmp DL,CPU16x4k_iIOStart
  jae LAS16_Trap

LAS16_Continue:

  and AL,AH
  mov EBP,[ESI]Chip.pNextChip

  mov [ESI]CPU65xx.bAccu,AL
  mov byte ptr ([ESI]CPU65xx.iXReg),AL

  mov byte ptr ([ESI]CPU65xx.iSP),AL
  mov byte ptr ([ESI]CPU65xx.iNCFlag[0]),AL

  mov [ESI]CPU65xx.bZVal,AL
  mov ESI,EBP

  ;two
  ;clocks

  ;unpaired
  jmp [EBP]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// HLTn - Halt CPU (no flags), no addressing
//
//  #  addres R/W description                                     proc
// --- ------ --- ----------------------------------------------- ---------
//  2+  ----   -  endless loop                                    HLTn2

proc(HLTn2) index(0)

  mov ESI,[ESI]Chip.pNextChip
  ;nop

  ;AGI
  ;stall

  ;two
  ;clocks

  ;unpaired
  jmp [ESI]Chip.pfnOnClock

endp


////////////////////////////////////////////////////////////////////////////////
// separate trap handlers prevent cache contamination

May2_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je May2_Adjust

May2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp May2_Continue

May2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae May2_Trap
  mov AL,[EBX]
  jmp May2_Continue

May3_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  push EAX
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je May3_Adjust

May3_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov BL,AL
  pop EAX
  mov AH,BL
  jmp May3_Continue

May3_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae May3_Trap
  pop EAX
  mov AH,[EBX]
  jmp May3_Continue

Mix2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Mix2_Adjust

Mix2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Mix2_Continue

Mix2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Mix2_Trap
  mov AL,[EBX]
  jmp Mix2_Continue

Mix3_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EAX
  jmp Mix3_Continue

Mix4_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Mix4_Continue

Mix5_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Mix5_Continue

Miy2_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je Miy2_Adjust

Miy2_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Miy2_Continue

Miy2_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae Miy2_Trap
  mov AL,[EBX]
  jmp Miy2_Continue

Miy3_Trap:

  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Miy3_Continue

Miy4_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov BL,AL
  pop EAX
  mov AH,BL
  jmp Miy4_Continue

Miy5_Trap:

  and EDX,0x000000FF
  mov [EBP]CPU65xx.iRWAddress,ECX
  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp Miy5_Continue

NOPb_Special:

  sub EBX,[EBP]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je NOPb_Adjust

NOPb_Trap:

  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp NOPb_Continue

NOPb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[EBP]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [EBP]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [EBP]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae NOPb_Trap
  jmp NOPb_Continue

NOP8_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov [ESI]CPU65xx.bAccu,AL
  jmp NOP8_Continue

NOP16_Trap:

  mov [EBP]CPU65xx.iRWAddress,ECX
  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp NOP16_Continue

LAX8_Trap:

  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov [ESI]CPU65xx.bAccu,AL
  jmp LAX8_Continue

LAX16_Trap:

  mov [EBP]CPU65xx.iRWAddress,ECX
  mov ECX,[EBP]CPU16x4k.apObject[EDX*4]
  call [EBP]CPU16x4k.abpfnOnRead[EDX*4]
  jmp LAX16_Continue

ALRb_Special:

  push EAX
  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je ALRb_Adjust

ALRb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp ALRb_Continue

ALRb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae ALRb_Trap
  pop EAX
  mov AL,[EBX]
  jmp ALRb_Continue

ARRb_Special:

  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je ARRb_Adjust

ARRb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,[ESI]CPU65xx.bAccu
  jmp ARRb_Continue

ARRb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae ARRb_Trap
  mov AL,[EBX]
  mov AH,[ESI]CPU65xx.bAccu
  jmp ARRb_Continue

ANEb_Special:

  push EAX
  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je ANEb_Adjust

ANEb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp ANEb_Continue

ANEb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae ANEb_Trap
  pop EAX
  mov AL,[EBX]
  jmp ANEb_Continue

LXAb_Special:

  sub EBX,[ESI]CPU65xx.pbPCBase
  and EDX,0x000000FF
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je LXAb_Adjust

LXAb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,[ESI]CPU65xx.bAccu
  jmp LXAb_Continue

LXAb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae LXAb_Trap
  mov AL,[EBX]
  mov AH,[ESI]CPU65xx.bAccu
  jmp LXAb_Continue

SBXb_Special:

  push EAX
  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je SBXb_Adjust

SBXb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  mov AH,AL
  pop ECX
  mov AL,CL
  jmp SBXb_Continue

SBXb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae SBXb_Trap
  pop EAX
  mov AH,[EBX]
  jmp SBXb_Continue

ANCb_Special:

  push EAX
  and EDX,0x000000FF
  sub EBX,[ESI]CPU65xx.pbPCBase
  and EBX,0x0000FFFF
  mov [ESI]CPU65xx.iRWAddress,EBX
  test EBX,4096-1
  je ANCb_Adjust

ANCb_Trap:

  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp ANCb_Continue

ANCb_Adjust:

  mov EAX,EBX
  shr EAX,12
  inc EBX
  mov EAX,[ESI]CPU16x4k.apbReadMapping[EAX*4]
  add EBX,EAX
  mov [ESI]CPU65xx.pbPCPtr,EBX
  dec EBX
  mov [ESI]CPU65xx.pbPCBase,EAX
  mov DL,[EBX+65536]
  cmp DL,CPU16x4k_iIOStart
  jae ANCb_Trap
  pop EAX
  mov AL,[EBX]
  jmp ANCb_Continue

LAS16_Trap:

  push EAX
  and EDX,0x000000FF
  mov [ESI]CPU65xx.iRWAddress,ECX
  mov ECX,[ESI]CPU16x4k.apObject[EDX*4]
  call [ESI]CPU16x4k.abpfnOnRead[EDX*4]
  pop EBX
  mov AH,BH
  jmp LAS16_Continue

}}
