/***************************************************************************
 *                                                                         *
 *    LIBDSK: General floppy and diskimage access library                  *
 *    Copyright (C) 2002  John Elliott <jce@seasip.demon.co.uk>            *
 *                                                                         *
 *    This library is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU Library General Public          *
 *    License as published by the Free Software Foundation; either         *
 *    version 2 of the License, or (at your option) any later version.     *
 *                                                                         *
 *    This library is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    *
 *    Library General Public License for more details.                     *
 *                                                                         *
 *    You should have received a copy of the GNU Library General Public    *
 *    License along with this library; if not, write to the Free           *
 *    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,      *
 *    MA 02111-1307, USA                                                   *
 *                                                                         *
 ***************************************************************************/

/* RPC functions: Server side */

#include "drvi.h"
#include "rpcfuncs.h"

/* Decode an RPC packet and execute it.
 * If nRefCount is nonzero, it is increased each time a dsk_open or dsk_creat 
 * succeeds, and decreased when a dsk_close succeeds. This allows a server 
 * program to terminate automatically when the reference count reaches 0. 
 *
 * The only RPC codes that are currently obeyed are those sent by drvwin32.c
 * through rpccli.c
 */
LDPUBLIC32 dsk_err_t LDPUBLIC16 dsk_rpc_server(unsigned char *input,  int inp_len,
	   		unsigned char *output, int *out_len, int *nRefCount)
	{
	dsk_err_t err, err2;
	int16 function;
	DSK_DRIVER *pDriver;
	DSK_GEOMETRY geom;
	DSK_FORMAT fmt, *pfmt;
	int32 nd, int1, int2, int3;
	int16 filler;
	char  *str1, *str2, *str3;
	unsigned int nDriver, n;
	unsigned char secbuf[8192], *pbuf;
	unsigned char status;

	err = dsk_unpack_i16(&input, &inp_len, &function); if (err) return err;
	switch(function)
		{
		case RPC_DSK_OPEN:
				err = dsk_unpack_string(&input, &inp_len, &str1); if (err) return err;
				err = dsk_unpack_string(&input, &inp_len, &str2); if (err) return err;
				err = dsk_unpack_string(&input, &inp_len, &str3); if (err) return err;
				err2= dsk_open(&pDriver, str1, str2, str3);
				err = dsk_map_dtoi(pDriver, &nDriver);            if (err) return err;
				err = dsk_pack_err  (&output, out_len, err2);    if (err) return err;
				err = dsk_pack_i32  (&output, out_len, nDriver); if (err) return err;
				if (nRefCount && err2 == DSK_ERR_OK) ++(*nRefCount);
				return DSK_ERR_OK;
		case RPC_DSK_CREAT:
				err = dsk_unpack_string(&input, &inp_len, &str1); if (err) return err;
				err = dsk_unpack_string(&input, &inp_len, &str2); if (err) return err;
				err = dsk_unpack_string(&input, &inp_len, &str3); if (err) return err;
				err2= dsk_creat(&pDriver, str1, str2, str3);
				err = dsk_map_dtoi(pDriver, &nDriver);            if (err) return err;
				err = dsk_pack_err  (&output, out_len, err2);    if (err) return err;
				err = dsk_pack_i32  (&output, out_len, nDriver); if (err) return err;
				if (nRefCount && err2 == DSK_ERR_OK) ++(*nRefCount);
				return DSK_ERR_OK;
		case RPC_DSK_CLOSE:
				err = dsk_unpack_i32(&input, &inp_len, &nd);	  if (err) return err;	nDriver = (unsigned int)nd;
				err = dsk_map_itod(nDriver, &pDriver);			  if (err) return err;
				err2= dsk_close(&pDriver);
				err = dsk_pack_err(&output,  out_len, err2);	  if (err) return err;
				if (nRefCount) --(*nRefCount);
				return DSK_ERR_OK;
		case RPC_DSK_PREAD:
				err = dsk_unpack_i32 (&input, &inp_len, &nd);	  if (err) return err;	nDriver = (unsigned int)nd;
				err = dsk_unpack_geom(&input, &inp_len, &geom);	  if (err) return err;
				err = dsk_unpack_i32 (&input, &inp_len, &int1);	  if (err) return err;
				err = dsk_unpack_i32 (&input, &inp_len, &int2);	  if (err) return err;
				err = dsk_unpack_i32 (&input, &inp_len, &int3);	  if (err) return err;
				err = dsk_map_itod(nDriver, &pDriver);			  if (err) return err;
				err2= dsk_pread(pDriver, &geom, secbuf, (dsk_pcyl_t)int1, (dsk_phead_t)int2, (dsk_psect_t)int3);
				err = dsk_pack_err(&output, out_len, err2);	  if (err) return err;
				err = dsk_pack_bytes(&output, out_len, secbuf, geom.dg_secsize); if (err) return err;
				return DSK_ERR_OK;
		case RPC_DSK_PWRITE:
				err = dsk_unpack_i32 (&input, &inp_len, &nd);	  if (err) return err;	nDriver = (unsigned int)nd;
				err = dsk_unpack_geom(&input, &inp_len, &geom);	  if (err) return err;
				err = dsk_unpack_bytes(&input, &inp_len, &pbuf);  if (err) return err;
				err = dsk_unpack_i32 (&input, &inp_len, &int1);	  if (err) return err;
				err = dsk_unpack_i32 (&input, &inp_len, &int2);	  if (err) return err;
				err = dsk_unpack_i32 (&input, &inp_len, &int3);	  if (err) return err;
				err = dsk_map_itod(nDriver, &pDriver);			  if (err) return err;
				err2= dsk_pwrite(pDriver, &geom, pbuf, (dsk_pcyl_t)int1, (dsk_phead_t)int2, (dsk_psect_t)int3);
				err = dsk_pack_err(&output, out_len, err2);	  if (err) return err;
				return DSK_ERR_OK;

		case RPC_DSK_PFORMAT:
				pfmt = (DSK_FORMAT *)secbuf;
				err = dsk_unpack_i32 (&input, &inp_len, &nd);	  if (err) return err;	nDriver = (unsigned int)nd;
				err = dsk_unpack_geom(&input, &inp_len, &geom);	  if (err) return err;
				err = dsk_unpack_i32 (&input, &inp_len, &int1);	  if (err) return err;
				err = dsk_unpack_i32 (&input, &inp_len, &int2);	  if (err) return err;
				for (n = 0; n < geom.dg_sectors; n++)
					{
					err = dsk_pack_format(&input, &inp_len, &pfmt[n]); if (err) return err;
					}
				err = dsk_unpack_i16 (&input, &inp_len, &filler);  if (err) return err;
				err = dsk_map_itod(nDriver, &pDriver);			  if (err) return err;
				err2= dsk_pformat(pDriver, &geom, (dsk_pcyl_t)int1, (dsk_phead_t)int2, pfmt, (unsigned char)filler);
				err = dsk_pack_err(&output, out_len, err2);	  if (err) return err;
				err = dsk_pack_geom(&output, out_len, &geom); if (err) return err;
				return DSK_ERR_OK;

		case RPC_DSK_PSECID:
				err = dsk_unpack_i32 (&input, &inp_len, &nd);	  if (err) return err;	nDriver = (unsigned int)nd;
				err = dsk_unpack_geom(&input, &inp_len, &geom);	  if (err) return err;
				err = dsk_unpack_i32 (&input, &inp_len, &int1);	  if (err) return err;
				err = dsk_unpack_i32 (&input, &inp_len, &int2);	  if (err) return err;
				err = dsk_map_itod(nDriver, &pDriver);			  if (err) return err;
				err2= dsk_psecid(pDriver, &geom, (dsk_pcyl_t)int1, (dsk_phead_t)int2, &fmt);
				err = dsk_pack_err(&output, out_len, err2);	  if (err) return err;
				err = dsk_pack_format(&output, out_len, &fmt); if (err) return err;
				return DSK_ERR_OK;

		case RPC_DSK_DRIVE_STATUS:
				err = dsk_unpack_i32 (&input, &inp_len, &nd);	  if (err) return err;	nDriver = (unsigned int)nd;
				err = dsk_unpack_geom(&input, &inp_len, &geom);	  if (err) return err;
				err = dsk_unpack_i32 (&input, &inp_len, &int1);	  if (err) return err;
				err = dsk_map_itod(nDriver, &pDriver);			  if (err) return err;
				err2= dsk_drive_status(pDriver, &geom, (dsk_pcyl_t)int1, &status);
				err = dsk_pack_err(&output, out_len, err2);	  if (err) return err;
				err = dsk_pack_i16(&output, out_len, status); if (err) return err;

		default: return DSK_ERR_RPC;
		}

	}



